<?php
require_once 'config/config.php';
require_once 'classes/Database.php';

$db = Database::getInstance();

// Step 1: Create proper user_searches table
echo "<h2>✅ Step 1: Creating user_searches Table</h2>";

try {
    $db->query("
        CREATE TABLE IF NOT EXISTS user_searches (
            id INT AUTO_INCREMENT PRIMARY KEY,
            user_id INT NOT NULL,
            search_query VARCHAR(255) NOT NULL,
            search_type VARCHAR(50),
            results_found INT DEFAULT 0,
            device_type VARCHAR(50) DEFAULT 'Unknown',
            ip_address VARCHAR(45),
            cost DECIMAL(10, 2) DEFAULT 0,
            status VARCHAR(50) DEFAULT 'PENDING',
            result_data LONGTEXT,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
            INDEX idx_user_id (user_id),
            INDEX idx_created_at (created_at),
            INDEX idx_status (status)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
    ");
    echo "✅ user_searches table created successfully!<br><br>";
} catch (Exception $e) {
    echo "⚠️ Table may already exist: " . $e->getMessage() . "<br><br>";
}

// Step 2: Check for existing search data in other tables
echo "<h2>📊 Step 2: Checking Existing Search Data</h2>";

$tables = $db->fetchAll("SHOW TABLES");
$search_tables = [];

foreach ($tables as $table) {
    $table_name = array_values($table)[0];
    
    // Look for search-related tables
    if (stripos($table_name, 'search') !== false || 
        stripos($table_name, 'mobile') !== false ||
        stripos($table_name, 'result') !== false) {
        
        try {
            $count = $db->fetch("SELECT COUNT(*) as cnt FROM `$table_name`");
            if ($count['cnt'] > 0) {
                $search_tables[$table_name] = $count['cnt'];
                echo "✅ Found: <strong>$table_name</strong> - " . $count['cnt'] . " records<br>";
            }
        } catch (Exception $e) {
            // Skip tables that error
        }
    }
}

echo "<br>";

// Step 3: Migrate data from existing tables
echo "<h2>🔄 Step 3: Migrating Search Data</h2>";

$migrated_count = 0;

// Check if searches table has data
try {
    $existing_searches = $db->fetchAll("SELECT * FROM searches WHERE user_id IS NOT NULL");
    
    if (count($existing_searches) > 0) {
        echo "Found " . count($existing_searches) . " searches to migrate<br>";
        
        foreach ($existing_searches as $search) {
            try {
                $db->insert('user_searches', [
                    'user_id' => $search['user_id'],
                    'search_query' => $search['search_query'],
                    'device_type' => $search['device_type'] ?? 'Unknown',
                    'cost' => $search['cost'] ?? 0,
                    'status' => $search['status'] ?? 'SUCCESS',
                    'created_at' => $search['created_at'] ?? date('Y-m-d H:i:s')
                ]);
                $migrated_count++;
            } catch (Exception $e) {
                // Skip duplicates
            }
        }
        echo "✅ Migrated " . $migrated_count . " searches<br><br>";
    }
} catch (Exception $e) {
    echo "ℹ️ No searches table or data to migrate<br>";
}

// Step 4: Fetch and display user searches
echo "<h2>📋 Step 4: User Searches Summary</h2>";

$users_with_searches = $db->fetchAll("
    SELECT u.id, u.name, u.email, COUNT(us.id) as search_count
    FROM users u
    LEFT JOIN user_searches us ON u.id = us.user_id
    GROUP BY u.id
    HAVING search_count > 0
    ORDER BY search_count DESC
    LIMIT 10
");

if (count($users_with_searches) > 0) {
    echo "<table border='1' cellpadding='10' style='width: 100%;'>";
    echo "<tr><th>User ID</th><th>Name</th><th>Email</th><th>Search Count</th></tr>";
    foreach ($users_with_searches as $user) {
        echo "<tr>";
        echo "<td>" . $user['id'] . "</td>";
        echo "<td>" . $user['name'] . "</td>";
        echo "<td>" . $user['email'] . "</td>";
        echo "<td>" . $user['search_count'] . "</td>";
        echo "</tr>";
    }
    echo "</table>";
} else {
    echo "No users with searches found yet<br>";
}

echo "<br>";

// Step 5: Show user 83 searches
echo "<h2>👤 Step 5: User #83 (Sandesh) Searches</h2>";

$user_83_searches = $db->fetchAll("SELECT * FROM user_searches WHERE user_id = 83 ORDER BY created_at DESC");

if (count($user_83_searches) > 0) {
    echo "Total searches: " . count($user_83_searches) . "<br><br>";
    echo "<table border='1' cellpadding='10' style='width: 100%;'>";
    echo "<tr><th>Query</th><th>Type</th><th>Device</th><th>Cost</th><th>Status</th><th>Date</th></tr>";
    foreach ($user_83_searches as $search) {
        echo "<tr>";
        echo "<td>" . htmlspecialchars($search['search_query']) . "</td>";
        echo "<td>" . $search['search_type'] . "</td>";
        echo "<td>" . $search['device_type'] . "</td>";
        echo "<td>₹" . $search['cost'] . "</td>";
        echo "<td><strong>" . $search['status'] . "</strong></td>";
        echo "<td>" . date('M d, g:i A', strtotime($search['created_at'])) . "</td>";
        echo "</tr>";
    }
    echo "</table>";
} else {
    echo "No searches for user 83<br>";
}

echo "<br>";

// Step 6: Summary
echo "<h2>✅ Migration Complete!</h2>";
echo "<p>Total records migrated: <strong>$migrated_count</strong></p>";
echo "<p><a href='admin/dashboard.php?page=search'>Go to Admin Search Page</a></p>";

?>
