<?php
/**
 * Get user's wallet transactions
 * Endpoint: /api/get_transactions.php
 * Query Parameters:
 *   - type: 'credit', 'debit', or empty for all
 *   - sort: 'latest', 'oldest', 'highest', 'lowest'
 */

require_once '../config/config.php';
require_once '../classes/Database.php';
require_once '../classes/Auth.php';

header('Content-Type: application/json; charset=utf-8');

$db = Database::getInstance();
$auth = new Auth($db);

// Check authentication
if (!$auth->isLoggedIn()) {
    http_response_code(401);
    echo json_encode([
        'success' => false,
        'message' => 'Unauthorized'
    ]);
    exit;
}

$currentUser = $auth->getCurrentUser();
$userId = $currentUser['id'];

// Get filter parameters
$typeFilter = $_GET['type'] ?? '';
$sortBy = $_GET['sort'] ?? 'latest';
$limit = (int)($_GET['limit'] ?? 100);

// Build query
$query = "SELECT id, user_id, type, amount, description, balance_after, created_at 
          FROM transactions 
          WHERE user_id = ?";

$params = [$userId];

// Add type filter
if (!empty($typeFilter) && in_array($typeFilter, ['credit', 'debit'])) {
    $query .= " AND type = ?";
    $params[] = $typeFilter;
}

// Add sorting
switch ($sortBy) {
    case 'oldest':
        $query .= " ORDER BY created_at ASC";
        break;
    case 'highest':
        $query .= " ORDER BY amount DESC";
        break;
    case 'lowest':
        $query .= " ORDER BY amount ASC";
        break;
    case 'latest':
    default:
        $query .= " ORDER BY created_at DESC";
        break;
}

$query .= " LIMIT " . $limit;

try {
    $transactions = $db->fetchAll($query, $params);
    
    echo json_encode([
        'success' => true,
        'transactions' => $transactions,
        'count' => count($transactions),
        'filter' => $typeFilter,
        'sort' => $sortBy
    ]);
} catch (Exception $e) {
    error_log('Error fetching transactions: ' . $e->getMessage());
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => 'Failed to fetch transactions: ' . $e->getMessage()
    ]);
}
?>
