<?php
require_once '../config/config.php';
require_once '../classes/Database.php';
require_once '../classes/Auth.php';

$db = Database::getInstance();
$auth = new Auth($db);

if (!$auth->isLoggedIn() || !$auth->isAdmin()) {
    header('Location: ../login.php');
    exit;
}

// Handle sending reply
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
    header('Content-Type: application/json');
    
    if ($_POST['action'] === 'send_reply') {
        $userId = intval($_POST['user_id'] ?? 0);
        $message = trim($_POST['message'] ?? '');
        
        if (empty($message) || $userId <= 0) {
            echo json_encode(['success' => false, 'message' => 'Invalid data']);
            exit;
        }
        
        $result = $db->insert('admin_chats', [
            'user_id' => $userId,
            'admin_id' => $_SESSION['user_id'],
            'message' => $message,
            'sender_type' => 'admin',
            'created_at' => date('Y-m-d H:i:s')
        ]);
        
        if ($result) {
            echo json_encode(['success' => true]);
        } else {
            echo json_encode(['success' => false, 'message' => 'Failed to send']);
        }
        exit;
    }
    
    if ($_POST['action'] === 'get_conversations') {
        // Get all unique users with unread messages
        $conversations = $db->fetchAll("
            SELECT DISTINCT u.id, u.name, u.email, u.custom_code,
                   (SELECT COUNT(*) FROM admin_chats ac WHERE ac.user_id = u.id AND ac.sender_type = 'admin' AND ac.is_read = 0) as unread,
                   (SELECT message FROM admin_chats ac WHERE ac.user_id = u.id ORDER BY ac.id DESC LIMIT 1) as last_message,
                   (SELECT created_at FROM admin_chats ac WHERE ac.user_id = u.id ORDER BY ac.id DESC LIMIT 1) as last_time
            FROM admin_chats ac
            JOIN users u ON u.id = ac.user_id
            ORDER BY ac.id DESC
        ");
        
        echo json_encode(['success' => true, 'conversations' => $conversations]);
        exit;
    }
    
    if ($_POST['action'] === 'get_user_messages') {
        $userId = intval($_POST['user_id'] ?? 0);
        
        if ($userId <= 0) {
            echo json_encode(['success' => false]);
            exit;
        }
        
        $messages = $db->fetchAll("
            SELECT id, message, sender_type, created_at
            FROM admin_chats
            WHERE user_id = ?
            ORDER BY id ASC
            LIMIT 100
        ", [$userId]);
        
        echo json_encode(['success' => true, 'messages' => $messages]);
        exit;
    }
}

$currentUser = $auth->getCurrentUser();
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Admin Chat Management</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <script src="https://unpkg.com/feather-icons"></script>
    <style>
        body {
            background: #f5f6f8;
            font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, sans-serif;
        }

        .chat-layout {
            display: flex;
            height: 100vh;
            background: white;
        }

        .conversations-list {
            width: 300px;
            border-right: 1px solid #e0e0e0;
            overflow-y: auto;
            background: #fafafa;
        }

        .conversation-item {
            padding: 12px;
            border-bottom: 1px solid #e0e0e0;
            cursor: pointer;
            transition: 0.2s;
        }

        .conversation-item:hover {
            background: #f0f0f0;
        }

        .conversation-item.active {
            background: #e3f2fd;
            border-left: 4px solid #667eea;
        }

        .conversation-item-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 4px;
        }

        .conversation-item-name {
            font-weight: 600;
            color: #333;
            font-size: 14px;
        }

        .conversation-item-unread {
            background: #667eea;
            color: white;
            border-radius: 12px;
            padding: 2px 8px;
            font-size: 11px;
            font-weight: 700;
        }

        .conversation-item-preview {
            font-size: 12px;
            color: #999;
            white-space: nowrap;
            overflow: hidden;
            text-overflow: ellipsis;
        }

        .messages-area {
            flex: 1;
            display: flex;
            flex-direction: column;
            background: white;
        }

        .messages-header {
            padding: 16px;
            border-bottom: 1px solid #e0e0e0;
            background: white;
        }

        .messages-header h5 {
            margin: 0;
            font-weight: 700;
        }

        .messages-header small {
            color: #999;
        }

        .messages-content {
            flex: 1;
            overflow-y: auto;
            padding: 16px;
        }

        .message {
            margin-bottom: 12px;
            display: flex;
            align-items: flex-end;
        }

        .message.admin {
            justify-content: flex-end;
        }

        .message-bubble {
            max-width: 60%;
            padding: 10px 14px;
            border-radius: 12px;
            word-wrap: break-word;
            font-size: 14px;
        }

        .message-bubble.user {
            background: #e8e8f0;
            color: #333;
        }

        .message-bubble.admin {
            background: #667eea;
            color: white;
        }

        .message-time {
            font-size: 11px;
            color: #999;
            margin: 0 8px;
        }

        .reply-area {
            padding: 16px;
            border-top: 1px solid #e0e0e0;
            display: flex;
            gap: 10px;
        }

        .reply-area input {
            flex: 1;
            border: 2px solid #e0e0e0;
            border-radius: 20px;
            padding: 10px 14px;
            outline: none;
        }

        .reply-area input:focus {
            border-color: #667eea;
        }

        .reply-area button {
            background: #667eea;
            color: white;
            border: none;
            border-radius: 50%;
            width: 40px;
            height: 40px;
            display: flex;
            align-items: center;
            justify-content: center;
            cursor: pointer;
        }

        .empty-state {
            display: flex;
            flex-direction: column;
            align-items: center;
            justify-content: center;
            height: 100%;
            color: #999;
        }

        @media (max-width: 768px) {
            .conversations-list {
                width: 100%;
            }
            .messages-area {
                display: none;
            }
            .conversations-list.show-messages {
                width: 0;
            }
            .messages-area.show-messages {
                display: flex;
            }
        }
    </style>
</head>
<body>

<div class="chat-layout">
    <!-- Conversations List -->
    <div class="conversations-list" id="conversationsList">
        <div style="padding: 16px; border-bottom: 1px solid #e0e0e0;">
            <h6 style="margin: 0;">💬 Messages</h6>
        </div>
        <div id="conversationsContainer" style="padding: 8px;">
            <p style="text-align: center; color: #999; padding: 20px 0;">Loading...</p>
        </div>
    </div>

    <!-- Messages Area -->
    <div class="messages-area">
        <div class="empty-state">
            <i data-feather="message-circle" style="width: 48px; height: 48px; color: #ccc; margin-bottom: 16px;"></i>
            <p>Select a conversation to start chatting</p>
        </div>

        <!-- Messages Content (hidden by default) -->
        <div id="messagesContent" style="display: none; flex: 1; display: flex; flex-direction: column;">
            <div class="messages-header">
                <h5 id="currentUserName">Select a user</h5>
                <small id="currentUserEmail"></small>
            </div>

            <div class="messages-content" id="messagesContainer"></div>

            <div class="reply-area">
                <input type="text" id="replyInput" placeholder="Type your reply...">
                <button onclick="sendReply()">
                    <i data-feather="send" style="width: 18px; height: 18px;"></i>
                </button>
            </div>
        </div>
    </div>
</div>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
<script>
    feather.replace();

    let currentUserId = null;
    let pollInterval;

    // Load conversations on startup
    loadConversations();

    // Poll for new conversations every 3 seconds
    pollInterval = setInterval(loadConversations, 3000);

    async function loadConversations() {
        try {
            const formData = new FormData();
            formData.append('action', 'get_conversations');

            const response = await fetch(window.location.href, {
                method: 'POST',
                body: formData
            });

            const data = await response.json();

            if (data.success) {
                renderConversations(data.conversations);
            }
        } catch (error) {
            console.error('Error loading conversations:', error);
        }
    }

    function renderConversations(conversations) {
        const container = document.getElementById('conversationsContainer');

        if (conversations.length === 0) {
            container.innerHTML = '<p style="text-align: center; color: #999; padding: 20px 0;">No conversations</p>';
            return;
        }

        let html = '';
        conversations.forEach(conv => {
            const time = new Date(conv.last_time).toLocaleTimeString('en-US', { 
                hour: '2-digit', 
                minute: '2-digit' 
            });

            html += `
                <div class="conversation-item" onclick="selectConversation(${conv.id}, '${escapeHtml(conv.name)}', '${escapeHtml(conv.email)}')">
                    <div class="conversation-item-header">
                        <div class="conversation-item-name">${escapeHtml(conv.name)}</div>
                        ${conv.unread > 0 ? `<div class="conversation-item-unread">${conv.unread}</div>` : ''}
                    </div>
                    <div class="conversation-item-preview">${escapeHtml(conv.last_message || 'No messages')} • ${time}</div>
                </div>
            `;
        });

        container.innerHTML = html;
    }

    async function selectConversation(userId, name, email) {
        currentUserId = userId;

        // Update header
        document.getElementById('currentUserName').textContent = name;
        document.getElementById('currentUserEmail').textContent = email;

        // Show messages area
        document.getElementById('messagesContent').style.display = 'flex';

        // Load messages
        loadMessages();

        // Clear input
        document.getElementById('replyInput').value = '';
        document.getElementById('replyInput').focus();
    }

    async function loadMessages() {
        if (!currentUserId) return;

        try {
            const formData = new FormData();
            formData.append('action', 'get_user_messages');
            formData.append('user_id', currentUserId);

            const response = await fetch(window.location.href, {
                method: 'POST',
                body: formData
            });

            const data = await response.json();

            if (data.success) {
                renderMessages(data.messages);
            }
        } catch (error) {
            console.error('Error loading messages:', error);
        }
    }

    function renderMessages(messages) {
        const container = document.getElementById('messagesContainer');

        if (messages.length === 0) {
            container.innerHTML = '<p style="text-align: center; color: #999; padding: 20px;">No messages yet</p>';
            return;
        }

        let html = '';
        messages.forEach(msg => {
            const isAdmin = msg.sender_type === 'admin';
            const time = new Date(msg.created_at).toLocaleTimeString('en-US', { 
                hour: '2-digit', 
                minute: '2-digit' 
            });

            html += `
                <div class="message ${isAdmin ? 'admin' : ''}">
                    ${!isAdmin ? '<div class="message-time">' + time + '</div>' : ''}
                    <div class="message-bubble ${msg.sender_type}">
                        ${escapeHtml(msg.message)}
                    </div>
                    ${isAdmin ? '<div class="message-time">' + time + '</div>' : ''}
                </div>
            `;
        });

        container.innerHTML = html;
        container.scrollTop = container.scrollHeight;
    }

    async function sendReply() {
        if (!currentUserId) return;

        const input = document.getElementById('replyInput');
        const message = input.value.trim();

        if (!message) return;

        input.value = '';

        try {
            const formData = new FormData();
            formData.append('action', 'send_reply');
            formData.append('user_id', currentUserId);
            formData.append('message', message);

            const response = await fetch(window.location.href, {
                method: 'POST',
                body: formData
            });

            const data = await response.json();

            if (data.success) {
                loadMessages();
            }
        } catch (error) {
            console.error('Error sending reply:', error);
        }
    }

    // Send on Enter
    document.addEventListener('DOMContentLoaded', function() {
        const replyInput = document.getElementById('replyInput');
        if (replyInput) {
            replyInput.addEventListener('keypress', function(e) {
                if (e.key === 'Enter') {
                    e.preventDefault();
                    sendReply();
                }
            });
        }
    });

    function escapeHtml(text) {
        const map = {
            '&': '&amp;',
            '<': '&lt;',
            '>': '&gt;',
            '"': '&quot;',
            "'": '&#039;'
        };
        return (text || '').replace(/[&<>"']/g, m => map[m]);
    }

    // Cleanup
    window.addEventListener('beforeunload', function() {
        if (pollInterval) clearInterval(pollInterval);
    });
</script>

</body>
</html>
