<?php
// user/dashboard.php (single-page AJAX dashboard)
// NOTE: replace the file at /extract/user/dashboard.php with this content.
error_reporting(E_ALL);
ini_set('display_errors', 1);
file_put_contents(__DIR__ . '/debug_log.txt', "POST DATA:\n" . print_r($_POST, true) . "\n", FILE_APPEND);


require_once '../config/config.php';
require_once '../classes/Database.php';
require_once '../classes/Auth.php';
require_once '../classes/User.php';
require_once '../classes/Search.php';

$db = Database::getInstance();
$auth = new Auth($db);

if (!$auth->isLoggedIn()) {
    header('Location: ../login.php');
    exit();
}
if ($auth->isAdmin()) {
    header('Location: ../admin/dashboard.php');
    exit();
}

$user = new User($db);
$search = new Search($db, $user);
$currentUser = $auth->getCurrentUser();

// Load initial data
$walletBalance = $user->getWalletBalance($currentUser['id']);
$permissions = $user->getPermissions($currentUser['id']);
$recentSearches = $user->getSearchHistory($currentUser['id'], 10);
$unreadMessages = $user->getUnreadMessages($currentUser['id']);
$adminSettings = $search->getAdminSettings();

// ---------- AJAX/POST handlers ----------
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // If AJAX request expects JSON response, we'll return JSON and exit.
    $action = $_POST['action'] ?? '';

    header('Content-Type: application/json; charset=utf-8');

    switch ($action) {
        // Mobile search (AJAX)
        case 'search_mobile':
            $mobile = trim($_POST['mobile'] ?? '');
            if (!preg_match('/^[6-9]\d{9}$/', $mobile)) {
                echo json_encode(['success' => false, 'message' => 'Invalid mobile number format']);
                exit;
            }
            $cost = (float)($adminSettings['mobile_search_cost'] ?? 1);
            // perform search using your Search class
            $result = $search->mobileSearch($currentUser['id'], $mobile);
            if ($result['success']) {
                // refresh wallet balance
                $balance = $user->getWalletBalance($currentUser['id']);
                // forward fallback flag if present
                echo json_encode([
                    'success' => true,
                    'data' => $result['data'],
                    'balance' => $balance,
                    'message' => $result['message'] ?? 'Search successful',
                    'fallback' => $result['fallback'] ?? false
                ]);
            } else {
                echo json_encode(['success' => false, 'message' => $result['message'] ?? 'Search failed']);
            }
            exit;
        // Aadhaar search (AJAX)
        case 'search_aadhaar':
            $aadhaar = trim($_POST['aadhaar'] ?? '');
            if (!preg_match('/^\d{12}$/', $aadhaar)) {
                echo json_encode(['success' => false, 'message' => 'Invalid Aadhaar number format']);
                exit;
            }
            $result = $search->aadhaarSearch($currentUser['id'], $aadhaar);
            if ($result['success']) {
                $balance = $user->getWalletBalance($currentUser['id']);
                echo json_encode([
                    'success' => true,
                    'data' => $result['data'],
                    'balance' => $balance,
                    'message' => $result['message'] ?? 'Search successful',
                    'fallback' => $result['fallback'] ?? false
                ]);
            } else {
                echo json_encode(['success' => false, 'message' => $result['message'] ?? 'Search failed']);
            }
            exit;
        // Update profile (AJAX)
        case 'update_profile':
            $name = trim($_POST['name'] ?? '');
            $email = trim($_POST['email'] ?? '');
            if ($name === '' || $email === '') {
                echo json_encode(['success' => false, 'message' => 'Name and email are required.']);
                exit;
            }
            $ok = $user->updateProfile($currentUser['id'], $name, $email);
            if ($ok) {
                // update session
                $_SESSION['user_name'] = $name;
                $_SESSION['user_email'] = $email;
                echo json_encode(['success' => true, 'message' => 'Profile updated', 'name' => $name, 'email' => $email]);
            } else {
                echo json_encode(['success' => false, 'message' => 'Failed to update profile']);
            }
            exit;
        // Add credits (AJAX)
        case 'add_credits':
            $amount = floatval($_POST['amount'] ?? 0);
            if ($amount <= 0) {
                echo json_encode(['success' => false, 'message' => 'Invalid amount']);
                exit;
            }
            $ok = $user->updateWalletBalance($currentUser['id'], $amount);
            if ($ok) {
                $balance = $user->getWalletBalance($currentUser['id']);
                echo json_encode(['success' => true, 'message' => 'Credits added', 'balance' => $balance]);
            } else {
                echo json_encode(['success' => false, 'message' => 'Failed to add credits']);
            }
            exit;
        // Submit support ticket (AJAX)
        case 'submit_ticket':
            $title = trim($_POST['title'] ?? '');
            $body = trim($_POST['body'] ?? '');
            if ($title === '' || $body === '') {
                echo json_encode(['success' => false, 'message' => 'Please fill all fields']);
                exit;
            }
            // Use your db wrapper insert if available, otherwise fallback to prepared query
            if (method_exists($db, 'insert')) {
                $ticketId = $db->insert('support_tickets', [
                    'user_id' => $currentUser['id'],
                    'title' => $title,
                    'body' => $body,
                    'status' => 'open',
                    'created_at' => date('Y-m-d H:i:s')
                ]);
                if ($ticketId) {
                    echo json_encode(['success' => true, 'message' => 'Ticket submitted', 'ticket_id' => $ticketId]);
                } else {
                    echo json_encode(['success' => false, 'message' => 'Failed to submit ticket']);
                }
            } else {
                // fallback
                $sql = "INSERT INTO support_tickets (user_id, title, body, status, created_at) VALUES (:uid, :title, :body, 'open', NOW())";
                $stmt = $db->prepare($sql);
                $ok = $stmt->execute([':uid' => $currentUser['id'], ':title' => $title, ':body' => $body]);
                if ($ok) echo json_encode(['success' => true, 'message' => 'Ticket submitted']);
                else echo json_encode(['success' => false, 'message' => 'Failed to submit ticket']);
            }
            exit;
        default:
            // Not an AJAX action we recognise — fallback: return error JSON
            echo json_encode(['success' => false, 'message' => 'Unknown action']);
            exit;
    }
}
// ---------- End AJAX handlers ----------

?><!DOCTYPE html>
<html lang="en">
<head>
<meta charset="utf-8">
<meta name="viewport" content="width=device-width,initial-scale=1">
<title>Dashboard - <?php echo APP_NAME; ?></title>

<link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
<script src="https://unpkg.com/feather-icons"></script>

<style>
/* your original styles (kept) */
:root {--primary-color:#007AFF;--secondary-color:#F2F2F7;--text-color:#1C1C1E;--light-gray:#F5F6F8;--border-color:#E5E5EA;--shadow-color:rgba(0,0,0,0.1);}
*{font-family:-apple-system,BlinkMacSystemFont,"Segoe UI",Roboto,"Helvetica Neue",Arial,sans-serif;}
body{background:var(--light-gray);color:var(--text-color);}
.sidebar{background:white;border-radius:16px;box-shadow:0 4px 20px var(--shadow-color);height:calc(100vh - 2rem);position:sticky;top:1rem;overflow-y:auto;}
.sidebar-header{padding:1.5rem;border-bottom:1px solid var(--border-color);text-align:center;}
.sidebar-menu{list-style:none;padding:0;margin:0;}
.sidebar-menu li{margin:0;}
.sidebar-menu a{display:flex;align-items:center;padding:1rem 1.5rem;color:var(--text-color);text-decoration:none;transition:all 0.3s ease;border-left:3px solid transparent;}
.sidebar-menu a:hover,.sidebar-menu a.active{background:rgba(0,122,255,0.1);border-left-color:var(--primary-color);color:var(--primary-color);}
.sidebar-menu i{margin-right:0.75rem;width:20px;height:20px;}
.main-content{padding:1rem;}
.top-bar{background:white;border-radius:16px;padding:1rem 1.5rem;margin-bottom:1.5rem;box-shadow:0 4px 20px var(--shadow-color);display:flex;justify-content:between;align-items:center;}
.wallet-badge{background:linear-gradient(135deg,#007AFF 0%,#0056CC 100%);color:white;padding:0.5rem 1rem;border-radius:20px;font-weight:600;font-size:0.9rem;}
.card{background:white;border-radius:16px;border:none;box-shadow:0 4px 20px var(--shadow-color);margin-bottom:1.5rem;}
.card-header{background:transparent;border-bottom:1px solid var(--border-color);padding:1.5rem;border-radius:16px 16px 0 0 !important;}
.card-body{padding:1.5rem;}
.btn-primary{background:linear-gradient(135deg,#007AFF 0%,#0056CC 100%);border:none;border-radius:12px;padding:10px 20px;font-weight:600;transition:all 0.3s ease;}
.btn-primary:hover{transform:translateY(-2px);box-shadow:0 8px 25px rgba(0,122,255,0.4);}
.form-control{border-radius:12px;border:2px solid var(--border-color);padding:0.75rem 1rem;transition:all 0.3s ease;}
.form-control:focus{border-color:var(--primary-color);box-shadow:0 0 0 0.25rem rgba(0,122,255,0.25);}
.section { display: none; } /* will show via JS */
.section.active { display: block; }
.small-muted { color: #6c757d; font-size: .9rem; }
</style>
</head>
<body>
<div class="container-fluid">
  <div class="row">
    <!-- Sidebar -->
    <div class="col-md-3 col-lg-2">
      <div class="sidebar">
        <div class="sidebar-header">
          <h5 class="mb-1" id="user-name"><?php echo htmlspecialchars($currentUser['name']); ?></h5>
          <small class="text-muted">User Dashboard</small>
        </div>

        <ul class="sidebar-menu">
          <li><a href="#" data-section="dashboard" class="nav-link active" onclick="showSection('dashboard')"><i data-feather="home"></i> Dashboard</a></li>
          <li><a href="#" data-section="mobile" class="nav-link" onclick="showSection('mobile')"><i data-feather="search"></i> Mobile Search</a></li>
          <li><a href="#" data-section="aadhaar" class="nav-link" onclick="showSection('aadhaar')"><i data-feather="credit-card"></i> Aadhaar Search</a></li>
          <li><a href="#" data-section="wallet" class="nav-link" onclick="showSection('wallet')"><i data-feather="dollar-sign"></i> Wallet <span class="badge bg-primary ms-auto" id="wallet-badge"><?php echo number_format($walletBalance, 2); ?></span></a></li>
          <li><a href="#" data-section="profile" class="nav-link" onclick="showSection('profile')"><i data-feather="user"></i> Profile</a></li>
          <li><a href="#" data-section="support" class="nav-link" onclick="showSection('support')"><i data-feather="message-circle"></i> Support <?php if (!empty($unreadMessages)): ?><span class="badge bg-danger ms-auto" id="unread-count"><?php echo count($unreadMessages); ?></span><?php endif; ?></a></li>
          <li><a href="../logout.php"><i data-feather="log-out"></i> Logout</a></li>
        </ul>
      </div>
    </div>

    <!-- Main Content -->
    <div class="col-md-9 col-lg-10">
      <div class="main-content">
        <div class="top-bar">
          <div>
            <h4 class="mb-0">Dashboard</h4>
            <small class="text-muted">Welcome back, <span id="topbar-name"><?php echo htmlspecialchars($currentUser['name']); ?></span></small>
          </div>
          <div class="d-flex align-items-center gap-3">
            <span class="wallet-badge" id="top-wallet">
              <i data-feather="inr-sign" style="width: 16px; height: 16px;" class="me-1"></i>
              <span id="wallet-amount">₹ <?php echo number_format($walletBalance, 2); ?></span> Credits
            </span>
            <small class="text-muted">For Every Search <strong> ₹10 </strong>will Deducted <span id="topbar-name"></span></small>
          
          </div>
        </div>

        <!-- ALERT PLACEHOLDER -->
        <div id="global-alert"></div>

        <!-- SECTION: Dashboard / Recent Searches summary -->
        <div id="dashboard-section" class="section active">
          <div class="card">
            <div class="card-header"><h5>Recent Searches</h5></div>
            <div class="card-body">
              <?php if (empty($recentSearches)): ?>
                <p class="text-muted">No searches yet</p>
              <?php else: ?>
                <div class="table-responsive">
                  <table class="table table-hover">
                    <thead><tr><th>Type</th><th>Query</th><th>Cost</th><th>Status</th><th>Date</th></tr></thead>
                    <tbody id="recent-searches-table">
                      <?php foreach ($recentSearches as $s): ?>
                        <tr>
                          <td><span class="badge bg-<?php echo $s['type'] === 'mobile' ? 'primary' : 'success'; ?>"><?php echo strtoupper($s['type']); ?></span></td>
                          <td><?php echo htmlspecialchars($s['query']); ?></td>
                          <td><?php echo number_format($s['cost'],1); ?></td>
                          <td><span class="badge bg-<?php echo $s['status'] === 'success' ? 'success' : 'danger'; ?>"><?php echo strtoupper($s['status']); ?></span></td>
                          <td><?php echo date('M j, Y H:i', strtotime($s['created_at'])); ?></td>
                        </tr>
                      <?php endforeach; ?>
                    </tbody>
                  </table>
                </div>
              <?php endif; ?>
            </div>
          </div>
        </div>

        <!-- SECTION: Mobile Search -->
        <div id="mobile-section" class="section">
          <div class="search-card">
            <h5>Mobile Lookup</h5>
            <form id="mobile-search-form" class="row g-3">
              <div class="col-md-8">
                <input type="text" id="mobile-input" class="form-control" placeholder="Enter 10-digit mobile number" pattern="[6-9]\d{9}" required>
              </div>
              <div class="col-md-4">
                <button type="submit" class="btn btn-primary w-100">Search</button>
              </div>
            </form>
            <div id="mobile-result" class="mt-4"></div>
          </div>
        </div>

        <!-- SECTION: Aadhaar Search -->
        <div id="aadhaar-section" class="section">
          <div class="search-card">
            <h5>Aadhaar Lookup</h5>
            <form id="aadhaar-search-form" class="row g-3">
              <div class="col-md-8">
                <input type="text" id="aadhaar-input" class="form-control" placeholder="Enter 12-digit Aadhaar number" pattern="\d{12}" required>
              </div>
              <div class="col-md-4">
                <button type="submit" class="btn btn-primary w-100">Search</button>
              </div>
            </form>
            <div id="aadhaar-result" class="mt-4"></div>
          </div>
        </div>

        <!-- SECTION: Wallet -->
        <div id="wallet-section" class="section">
          <div class="card">
            <div class="card-header"><h5>Wallet</h5></div>
            <div class="card-body">
              <p class="small-muted">Current balance:</p>
              <h3 id="wallet-display">₹ <?php echo number_format($walletBalance, 2); ?></h3>

              <hr>

              <h6>Add Credits</h6>
              <form id="add-credits-form" class="row g-3">
                <div class="col-md-8">
                  <input type="number" name="amount" id="credit-amount" class="form-control" step="0.01" min="1" placeholder="Amount to add" required>
                </div>
                <div class="col-md-4">
                  <button type="submit" class="btn btn-primary w-100">Add Credits</button>
                </div>
              </form>
            </div>
          </div>
        </div>

        <!-- SECTION: Profile -->
        <div id="profile-section" class="section">
          <div class="card">
            <div class="card-header"><h5>Profile</h5></div>
            <div class="card-body">
              <form id="profile-form">
                <div class="mb-3">
                  <label class="form-label">Full Name</label>
                  <input type="text" name="name" id="profile-name" class="form-control" required value="<?php echo htmlspecialchars($currentUser['name']); ?>">
                </div>
                <div class="mb-3">
                  <label class="form-label">Email</label>
                  <input type="email" name="email" id="profile-email" class="form-control" required value="<?php echo htmlspecialchars($currentUser['email']); ?>">
                </div>
                <div class="text-end">
                  <button type="submit" class="btn btn-primary">Save Profile</button>
                </div>
              </form>
            </div>
          </div>
        </div>

        <!-- SECTION: Support -->
        <div id="support-section" class="section">
          <div class="card">
            <div class="card-header"><h5>Support</h5></div>
            <div class="card-body">
              <form id="support-form">
                <div class="mb-3">
                  <label class="form-label">Title</label>
                  <input type="text" id="support-title" class="form-control" required>
                </div>
                <div class="mb-3">
                  <label class="form-label">Message</label>
                  <textarea id="support-body" class="form-control" rows="6" required></textarea>
                </div>
                <div class="text-end">
                  <button type="submit" class="btn btn-primary">Submit Ticket</button>
                </div>
              </form>

              <hr>

              <h6>Your recent tickets</h6>
              <div id="tickets-list" class="mt-3">
                <?php
                  // show last 5 support tickets by user (if table exists)
                  try {
                    $tickets = $db->fetchAll('SELECT id, title, status, created_at FROM support_tickets WHERE user_id = :uid ORDER BY created_at DESC LIMIT 5', [':uid' => $currentUser['id']]);
                    if ($tickets) {
                      foreach ($tickets as $t) {
                        echo '<div class="mb-2"><strong>'.htmlspecialchars($t['title']).'</strong><br><small class="text-muted">'.htmlspecialchars($t['status']).' • '.date('M j, Y H:i', strtotime($t['created_at'])).'</small></div>';
                      }
                    } else {
                      echo '<p class="text-muted">No tickets yet.</p>';
                    }
                  } catch (Exception $e) {
                    echo '<p class="text-muted">Tickets not available.</p>';
                  }
                ?>
              </div>
            </div>
          </div>
        </div>

      </div><!-- /.main-content -->
    </div>
  </div>
</div>

<!-- scripts -->
<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
<script>
feather.replace();

// simple section switcher
function showSection(name) {
  // hide all sections
  document.querySelectorAll('.section').forEach(s => s.classList.remove('active'));
  // remove active class from menu links
  document.querySelectorAll('.sidebar-menu .nav-link').forEach(a => a.classList.remove('active'));
  // show requested
  const el = document.getElementById(name + '-section');
  if (el) el.classList.add('active');
  // set active menu link
  const link = document.querySelector('.sidebar-menu a[data-section="'+name+'"]');
  if (link) link.classList.add('active');
}

// utility: show global alert
function showAlert(type, message, timeout=4000) {
  const alertBox = document.getElementById('global-alert');
  alertBox.innerHTML = `<div class="alert alert-${type} alert-dismissible fade show" role="alert">${message}<button type="button" class="btn-close" data-bs-dismiss="alert"></button></div>`;
  if (timeout>0) setTimeout(()=>{ alertBox.innerHTML = ''; }, timeout);
}

// -------- AJAX helpers --------
async function postForm(action, data) {
  data = data || {};
  data.action = action;
  const formData = new FormData();
  for (const k in data) {
    formData.append(k, data[k]);
  }
  const resp = await fetch(location.href, { method: 'POST', body: formData });
  const json = await resp.json();
  return json;
}

// Mobile search handler
document.getElementById('mobile-search-form').addEventListener('submit', async function(e){
  e.preventDefault();
  const mobile = document.getElementById('mobile-input').value.trim();
  if (!/^[6-9]\d{9}$/.test(mobile)) { showAlert('warning','Invalid mobile number'); return; }
  const btn = this.querySelector('button[type="submit"]');
  btn.disabled = true;
  btn.innerText = 'Searching...';
  try {
    const res = await postForm('search_mobile', { mobile });
    if (res.success) {
      renderMobileResult(res.data, res.fallback);
      // update wallet display
      if (res.balance !== undefined) updateWalletDisplay(res.balance);
      showAlert('success', res.message || 'Search successful');
      // optionally refresh recent searches table by prepending new row
      prependRecentSearch('mobile', mobile, res.data.cost || 0, 'success');
    } else {
      showAlert('danger', res.message || 'Search failed');
    }
  } catch (err) {
    showAlert('danger', 'Request failed');
  } finally {
    btn.disabled = false;
    btn.innerText = 'Search';
  }
});

// Aadhaar search handler
document.getElementById('aadhaar-search-form').addEventListener('submit', async function(e){
  e.preventDefault();
  const aadhaar = document.getElementById('aadhaar-input').value.trim();
  if (!/^\d{12}$/.test(aadhaar)) { showAlert('warning','Invalid Aadhaar number'); return; }
  const btn = this.querySelector('button[type="submit"]');
  btn.disabled = true;
  btn.innerText = 'Searching...';
  try {
    const res = await postForm('search_aadhaar', { aadhaar });
    if (res.success) {
      renderAadhaarResult(res.data, res.fallback);
      if (res.balance !== undefined) updateWalletDisplay(res.balance);
      showAlert('success', res.message || 'Search successful');
      prependRecentSearch('aadhaar', aadhaar, res.data.cost || 0, 'success');
    } else {
      showAlert('danger', res.message || 'Search failed');
    }
  } catch (err) {
    showAlert('danger', 'Request failed');
  } finally {
    btn.disabled = false;
    btn.innerText = 'Search';
  }
});

// profile update (AJAX)
document.getElementById('profile-form').addEventListener('submit', async function(e){
  e.preventDefault();
  const name = document.getElementById('profile-name').value.trim();
  const email = document.getElementById('profile-email').value.trim();
  if (!name || !email) { showAlert('warning','Name and email required'); return; }
  const res = await postForm('update_profile', { name, email });
  if (res.success) {
    // update UI names
    document.getElementById('user-name').innerText = res.name;
    document.getElementById('topbar-name').innerText = res.name;
    showAlert('success', res.message);
  } else showAlert('danger', res.message || 'Failed to update profile');
});

// add credits (AJAX)
document.getElementById('add-credits-form').addEventListener('submit', async function(e){
  e.preventDefault();
  const amount = parseFloat(document.getElementById('credit-amount').value);
  if (!amount || amount <= 0) { showAlert('warning','Enter a valid amount'); return; }
  const btn = this.querySelector('button[type="submit"]');
  btn.disabled = true; btn.innerText = 'Adding...';
  const res = await postForm('add_credits', { amount });
  if (res.success) {
    updateWalletDisplay(res.balance);
    showAlert('success', res.message || 'Credits added');
  } else showAlert('danger', res.message || 'Failed to add credits');
  btn.disabled = false; btn.innerText = 'Add Credits';
});

// submit support ticket (AJAX)
document.getElementById('support-form').addEventListener('submit', async function(e){
  e.preventDefault();
  const title = document.getElementById('support-title').value.trim();
  const body = document.getElementById('support-body').value.trim();
  if (!title || !body) { showAlert('warning','Please fill title and message'); return; }
  const btn = this.querySelector('button[type="submit"]');
  btn.disabled = true; btn.innerText = 'Submitting...';
  const res = await postForm('submit_ticket', { title, body });
  if (res.success) {
    showAlert('success', res.message || 'Ticket submitted');
    // prepend ticket to list (simple)
    const list = document.getElementById('tickets-list');
    const node = document.createElement('div');
    node.className = 'mb-2';
    node.innerHTML = `<strong>${escapeHtml(title)}</strong><br><small class="text-muted">open • just now</small>`;
    list.prepend(node);
    // clear form
    document.getElementById('support-title').value = '';
    document.getElementById('support-body').value = '';
  } else {
    showAlert('danger', res.message || 'Failed to submit ticket');
  }
  btn.disabled = false; btn.innerText = 'Submit Ticket';
});

// helper: render mobile result (expects object or array)
function renderMobileResult(data, fallback) {
  const box = document.getElementById('mobile-result');
  // data may be array or single object; adapt to your API format
  let html = '';
  if (Array.isArray(data)) {
    data.forEach(item => {
      html += mobileCardHtml(item, fallback);
    });
  } else if (typeof data === 'object') {
    // if your API returned wrapper { data: [...] } maybe second-level
    if (data.data && Array.isArray(data.data)) {
      data.data.forEach(item => html += mobileCardHtml(item, fallback));
    } else {
      html = mobileCardHtml(data, fallback);
    }
  } else {
    html = `<div class="alert alert-info">No result</div>`;
  }
  box.innerHTML = html;
}
function mobileCardHtml(item, fallback) {
  return `<div class="card search-result-card mb-3 p-3">
    ${fallback 
      ? "<div class='alert alert-warning p-2 mb-2'>⚠️ Data Source: Database Backup</div>"
      : "<div class='alert alert-success p-2 mb-2'>🟢 Data Source: LIVE API</div>"
    }
    <div><strong>Name:</strong> ${escapeHtml(item.name || '')}</div>
    <div><strong>Mobile:</strong> ${escapeHtml(item.mobile || '')}</div>
    <div><strong>Father's Name:</strong> ${escapeHtml(item.fname || '')}</div>
    <div><strong>Circle:</strong> ${escapeHtml(item.circle || '')}</div>
    <div><strong>Alt No:</strong> ${escapeHtml(item.alt || '')}</div>
    <div><strong>Aadhar Card:</strong> ${escapeHtml(item.id || '')}</div>
    <div><strong>Address:</strong> ${escapeHtml(item.address || '')}</div>
  </div>`;
}


function renderAadhaarResult(data, fallback) {
  const box = document.getElementById('aadhaar-result');
  let html = '';
  if (typeof data === 'object') {
    // show main fields
    html += '<div class="card search-result-card p-3">';
    html += (fallback 
      ? "<div class='alert alert-warning p-2 mb-2'>⚠️ Data Source: Database Backup</div>"
      : "<div class='alert alert-success p-2 mb-2'>🟢 Data Source: LIVE API</div>"
    );
    html += `<div><strong>Address:</strong> ${escapeHtml(data.address || '')}</div>`;
    html += `<div><strong>Home District:</strong> ${escapeHtml(data.homeDistName || data.homeDistrict || '')}</div>`;
    html += `<div><strong>Home State:</strong> ${escapeHtml(data.homeStateName || '')}</div>`;
    if (data.memberDetailsList && Array.isArray(data.memberDetailsList)) {
      html += '<hr><h6>Members</h6><ul>';
      data.memberDetailsList.forEach(m => {
        html += `<li>${escapeHtml(m.memberName || '')} — ${escapeHtml(m.releationship_name || '')} (${escapeHtml(m.memberId || '')})</li>`;
      });
      html += '</ul>';
    }
    html += '</div>';
  } else {
    html = '<div class="alert alert-info">No result</div>';
  }
  box.innerHTML = html;
}

function prependRecentSearch(type, query, cost, status) {
  try {
    const tbody = document.querySelector('#recent-searches-table');
    if (!tbody) return;
    const tr = document.createElement('tr');
    tr.innerHTML = `<td><span class="badge bg-${type==='mobile'?'primary':'success'}">${type.toUpperCase()}</span></td>
      <td>${escapeHtml(query)}</td>
      <td>${Number(cost || 0).toFixed(1)}</td>
      <td><span class="badge bg-${status==='success'?'success':'danger'}">${status.toUpperCase()}</span></td>
      <td>${new Date().toLocaleString()}</td>`;
    tbody.prepend(tr);
  } catch (e) { /* ignore */ }
}

function updateWalletDisplay(amount) {
  document.getElementById('wallet-display').innerText = '₹ ' + Number(amount).toFixed(2);
  document.getElementById('wallet-amount').innerText = Number(amount).toFixed(2);
  const badge = document.getElementById('wallet-badge');
  if (badge) badge.innerText = Number(amount).toFixed(2);
}

// small utility to escape html
function escapeHtml(str) {
  if (!str) return '';
  return String(str).replace(/&/g,'&amp;').replace(/</g,'&lt;').replace(/>/g,'&gt;').replace(/"/g,'&quot;');
}

// start on dashboard by default
showSection('dashboard');

</script>
</body>
</html>
