<?php
session_start();
require_once 'db.php';

// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    header('Location: index.php');
    exit();
}

$user_id = $_SESSION['user_id'];
$user_name = $_SESSION['user_name'];

// Get toast messages
$toast_success = isset($_SESSION['toast_success']) ? $_SESSION['toast_success'] : '';
$toast_error = isset($_SESSION['toast_error']) ? $_SESSION['toast_error'] : '';

// Clear toast messages
unset($_SESSION['toast_success']);
unset($_SESSION['toast_error']);
?>
<!DOCTYPE html>
<html lang="en">

<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0, maximum-scale=1.0, user-scalable=no">
    <meta name="apple-mobile-web-app-capable" content="yes">
    <meta name="apple-mobile-web-app-status-bar-style" content="default">
    <meta name="theme-color" content="#007AFF">
    <title>Mobile Search</title>
    <link rel="stylesheet" href="style.css?v=<?php echo time(); ?>">
    <style>
        .search-container {
            padding: 20px;
            padding-bottom: 80px;
        }

        .search-header {
            margin-bottom: 30px;
        }

        .search-title {
            font-size: 28px;
            font-weight: 700;
            color: var(--text-primary);
            margin-bottom: 8px;
        }

        .search-subtitle {
            font-size: 14px;
            color: var(--text-secondary);
        }

        .search-box {
            background: white;
            border-radius: 16px;
            padding: 24px;
            box-shadow: 0 2px 12px rgba(0, 0, 0, 0.08);
            margin-bottom: 24px;
        }

        .search-input-group {
            margin-bottom: 16px;
        }

        .search-input-label {
            display: block;
            font-size: 13px;
            font-weight: 600;
            color: var(--text-primary);
            margin-bottom: 8px;
        }

        .search-input {
            width: 100%;
            padding: 14px 16px;
            border: 2px solid var(--border-color);
            border-radius: 12px;
            font-size: 16px;
            font-family: -apple-system, BlinkMacSystemFont, "SF Pro Display", "SF Pro Text", system-ui, Helvetica, Arial, sans-serif;
            transition: all 0.2s ease;
            background: var(--bg-secondary);
        }

        .search-input:focus {
            outline: none;
            border-color: #007AFF;
            background: white;
        }

        .search-btn {
            width: 100%;
            padding: 16px;
            background: linear-gradient(135deg, #007AFF 0%, #0051D5 100%);
            color: white;
            border: none;
            border-radius: 12px;
            font-size: 16px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.2s ease;
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 8px;
        }

        .search-btn:active {
            transform: scale(0.98);
        }

        .search-btn.loading {
            opacity: 0.7;
            pointer-events: none;
        }

        .search-btn.loading::after {
            content: '';
            width: 16px;
            height: 16px;
            border: 2px solid white;
            border-top-color: transparent;
            border-radius: 50%;
            animation: spin 0.6s linear infinite;
        }

        @keyframes spin {
            to {
                transform: rotate(360deg);
            }
        }

        .result-card {
            background: white;
            border-radius: 16px;
            padding: 20px;
            box-shadow: 0 2px 12px rgba(0, 0, 0, 0.08);
            display: none;
            animation: slideUp 0.3s ease;
        }

        .result-card.show {
            display: block;
        }

        @keyframes slideUp {
            from {
                opacity: 0;
                transform: translateY(20px);
            }

            to {
                opacity: 1;
                transform: translateY(0);
            }
        }

        .result-header {
            display: flex;
            align-items: center;
            gap: 12px;
            margin-bottom: 20px;
            padding-bottom: 16px;
            border-bottom: 1px solid var(--border-color);
        }

        .result-icon {
            width: 48px;
            height: 48px;
            border-radius: 50%;
            background: linear-gradient(135deg, #34C759 0%, #30D158 100%);
            display: flex;
            align-items: center;
            justify-content: center;
            flex-shrink: 0;
        }

        .result-icon svg {
            width: 24px;
            height: 24px;
        }

        .result-title {
            font-size: 18px;
            font-weight: 700;
            color: var(--text-primary);
        }

        .result-row {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: 12px 0;
            border-bottom: 1px solid var(--border-color);
        }

        .result-row:last-child {
            border-bottom: none;
        }

        .result-label {
            font-size: 13px;
            color: var(--text-secondary);
            font-weight: 500;
        }

        .result-value {
            font-size: 14px;
            color: var(--text-primary);
            font-weight: 600;
            text-align: right;
        }

        .error-card {
            background: white;
            border-radius: 16px;
            padding: 32px 20px;
            box-shadow: 0 2px 12px rgba(0, 0, 0, 0.08);
            text-align: center;
            display: none;
        }

        .error-card.show {
            display: block;
            animation: slideUp 0.3s ease;
        }

        .error-icon {
            width: 64px;
            height: 64px;
            margin: 0 auto 16px;
            border-radius: 50%;
            background: rgba(255, 59, 48, 0.1);
            display: flex;
            align-items: center;
            justify-content: center;
        }

        .error-icon svg {
            width: 32px;
            height: 32px;
        }

        .error-title {
            font-size: 18px;
            font-weight: 700;
            color: var(--text-primary);
            margin-bottom: 8px;
        }

        .error-message {
            font-size: 14px;
            color: var(--text-secondary);
            line-height: 1.5;
        }

        .source-badge {
            display: inline-block;
            padding: 4px 12px;
            border-radius: 12px;
            font-size: 11px;
            font-weight: 600;
            margin-top: 12px;
        }

        .source-db {
            background: rgba(52, 199, 89, 0.1);
            color: #34C759;
        }

        .source-api {
            background: rgba(0, 122, 255, 0.1);
            color: #007AFF;
        }

        /* Voice Search Button */
        .voice-btn {
            width: 48px;
            height: 48px;
            border-radius: 12px;
            border: 2px solid #E5E5EA;
            background: white;
            display: flex;
            align-items: center;
            justify-content: center;
            cursor: pointer;
            transition: all 0.3s ease;
            flex-shrink: 0;
        }

        .voice-btn:hover {
            background: #F5F5F7;
            border-color: #007AFF;
        }

        .voice-btn svg {
            color: #007AFF;
            transition: all 0.3s ease;
        }

        .voice-btn.listening {
            background: #FF3B30;
            border-color: #FF3B30;
            animation: pulse 1.5s infinite;
        }

        .voice-btn.listening svg {
            color: white;
        }

        @keyframes pulse {

            0%,
            100% {
                transform: scale(1);
                box-shadow: 0 0 0 0 rgba(255, 59, 48, 0.7);
            }

            50% {
                transform: scale(1.05);
                box-shadow: 0 0 0 10px rgba(255, 59, 48, 0);
            }
        }
    </style>
</head>

<body>
    <div class="mobile-container">
        <div class="search-container">
            <!-- Header -->
            <div class="search-header">
                <h1 class="search-title">Mobile Search</h1>
                <p class="search-subtitle">Find mobile number information instantly</p>
            </div>

            <!-- Search Box -->
            <div class="search-box">
                <form id="searchForm">
                    <div class="search-input-group">
                        <label class="search-input-label">Mobile Number</label>
                        <div style="position: relative; display: flex; gap: 8px;">
                            <input type="tel" id="mobileInput" class="search-input"
                                placeholder="Enter 10-digit mobile number" maxlength="12" pattern="[0-9]{10}" required
                                style="flex: 1;">
                            <button type="button" id="voiceBtn" class="voice-btn" onclick="startVoiceSearch()"
                                title="Voice Search">
                                <svg width="20" height="20" viewBox="0 0 24 24" fill="currentColor" id="micIcon">
                                    <path
                                        d="M12 14c1.66 0 3-1.34 3-3V5c0-1.66-1.34-3-3-3S9 3.34 9 5v6c0 1.66 1.34 3 3 3z" />
                                    <path
                                        d="M17 11c0 2.76-2.24 5-5 5s-5-2.24-5-5H5c0 3.53 2.61 6.43 6 6.92V21h2v-3.08c3.39-.49 6-3.39 6-6.92h-2z" />
                                </svg>
                            </button>
                        </div>
                        <div id="voiceStatus" style="display: none; margin-top: 8px; font-size: 13px; color: #FF3B30;">
                        </div>
                    </div>
                    <button type="submit" class="search-btn" id="searchBtn">
                        <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor"
                            stroke-width="2">
                            <circle cx="11" cy="11" r="8" />
                            <path d="m21 21-4.35-4.35" />
                        </svg>
                        Search Mobile
                    </button>
                </form>
            </div>

            <!-- Aadhaar Search Section -->
            <div class="search-box" style="margin-top: 20px;">
                <div style="display: flex; align-items: center; gap: 8px; margin-bottom: 12px;">
                    <svg width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="#FF9500" stroke-width="2">
                        <rect x="3" y="4" width="18" height="16" rx="2" />
                        <path d="M7 8h10M7 12h10M7 16h6" />
                    </svg>
                    <h3 style="margin: 0; font-size: 18px; font-weight: 600; color: #FF9500;">Aadhaar Search</h3>
                </div>
                <form id="aadhaarForm" onsubmit="event.preventDefault(); searchAadhaar();">
                    <div class="search-input-group">
                        <label class="search-input-label">Aadhaar Number</label>
                        <input type="tel" id="aadhaarInput" class="search-input"
                            placeholder="Enter 12-digit Aadhaar number" maxlength="12" pattern="[0-9]{12}" required>
                    </div>
                    <button type="submit" class="search-btn" id="aadhaarBtn"
                        style="background: linear-gradient(135deg, #FF9500 0%, #FF5E3A 100%);">
                        <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor"
                            stroke-width="2">
                            <rect x="3" y="4" width="18" height="16" rx="2" />
                            <path d="M7 8h10M7 12h10M7 16h6" />
                        </svg>
                        Search Aadhaar
                    </button>
                </form>
            </div>

            <!-- Result Card -->
            <div class="result-card" id="resultCard">
                <div class="result-header">
                    <div class="result-icon">
                        <svg viewBox="0 0 24 24" fill="white">
                            <path
                                d="M20.01 15.38c-1.23 0-2.42-.2-3.53-.56-.35-.12-.74-.03-1.01.24l-1.57 1.97c-2.83-1.35-5.48-3.9-6.89-6.83l1.95-1.66c.27-.28.35-.67.24-1.02-.37-1.11-.56-2.3-.56-3.53 0-.54-.45-.99-.99-.99H4.19C3.65 3 3 3.24 3 3.99 3 13.28 10.73 21 20.01 21c.71 0 .99-.63.99-1.18v-3.45c0-.54-.45-.99-.99-.99z" />
                        </svg>
                    </div>
                    <div>
                        <div class="result-title" id="resultMobile"></div>
                        <span class="source-badge" id="sourceBadge"></span>
                    </div>
                </div>
                <div id="resultData"></div>
            </div>

            <!-- Error Card -->
            <div class="error-card" id="errorCard">
                <div class="error-icon">
                    <svg viewBox="0 0 24 24" fill="#FF3B30">
                        <path
                            d="M12 2C6.48 2 2 6.48 2 12s4.48 10 10 10 10-4.48 10-10S17.52 2 12 2zm1 15h-2v-2h2v2zm0-4h-2V7h2v6z" />
                    </svg>
                </div>
                <div class="error-title" id="errorTitle">Error</div>
                <div class="error-message" id="errorMessage"></div>
            </div>
        </div>
    </div>

    <!-- Bottom Navigation -->
    <?php include 'navbar.php'; ?>

    <script src="script.js"></script>
    <script>
        const searchForm = document.getElementById('searchForm');
        const mobileInput = document.getElementById('mobileInput');
        const searchBtn = document.getElementById('searchBtn');
        const resultCard = document.getElementById('resultCard');
        const errorCard = document.getElementById('errorCard');

        // Only allow numbers in input
        mobileInput.addEventListener('input', function (e) {
            this.value = this.value.replace(/[^0-9]/g, '');
        });

        searchForm.addEventListener('submit', async function (e) {
            e.preventDefault();

            const mobile = mobileInput.value.trim();

            // Validate mobile number
            if (mobile.length !== 10) {
                showError('Invalid Number', 'Please enter a valid 10-digit mobile number');
                return;
            }

            // Hide previous results
            resultCard.classList.remove('show');
            errorCard.classList.remove('show');

            // Show loading state
            searchBtn.classList.add('loading');
            searchBtn.textContent = 'Searching';

            try {
                const response = await fetch('api_fetch.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/x-www-form-urlencoded',
                    },
                    body: `mobile=${mobile}`
                });

                const data = await response.json();

                if (data.success) {
                    showResult(data.data, data.source);
                } else {
                    showError(data.error_title || 'Not Found', data.message || 'No data available for this number');
                }
            } catch (error) {
                showError('Connection Error', 'Unable to connect to server. Please try again.');
            } finally {
                // Reset button
                searchBtn.classList.remove('loading');
                searchBtn.innerHTML = `
                    <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                        <circle cx="11" cy="11" r="8"/>
                        <path d="m21 21-4.35-4.35"/>
                    </svg>
                    Search
                `;
            }
        });

        function showResult(data, source) {
            // Check if data is array (multiple records) or single object
            const records = Array.isArray(data) ? data : [data];
            const recordCount = records.length;

            // Clear previous results
            document.getElementById('resultData').innerHTML = '';

            // Show count if multiple records
            if (recordCount > 1) {
                document.getElementById('resultMobile').textContent = records[0].mobile;
                document.getElementById('sourceBadge').innerHTML = `
                    <span class="source-badge source-db">⚡ ${recordCount} Records Found</span>
                `;
            } else {
                document.getElementById('resultMobile').textContent = records[0].mobile;
                const sourceBadge = document.getElementById('sourceBadge');
                if (source === 'database') {
                    sourceBadge.textContent = '⚡ From Database';
                    sourceBadge.className = 'source-badge source-db';
                } else {
                    sourceBadge.textContent = '🌐 From API';
                    sourceBadge.className = 'source-badge source-api';
                }
            }

            let allResultsHTML = '';

            // Loop through each record
            records.forEach((record, index) => {
                let resultHTML = '';

                // Record header if multiple
                if (recordCount > 1) {
                    resultHTML += `
                        <div style="background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; padding: 8px 12px; border-radius: 8px; margin-bottom: 12px; font-size: 12px; font-weight: 600;">
                            📋 Record ${index + 1} of ${recordCount}
                        </div>
                    `;
                }

                // Name
                if (record.name && record.name !== 'N/A') {
                    resultHTML += `
                        <div class="result-row">
                            <span class="result-label">📛 Name</span>
                            <span class="result-value">${record.name}</span>
                        </div>
                    `;
                }

                // Father's Name
                if (record.fname && record.fname !== 'N/A') {
                    resultHTML += `
                        <div class="result-row">
                            <span class="result-label">👨 Father's Name</span>
                            <span class="result-value">${record.fname}</span>
                        </div>
                    `;
                }

                // Circle
                if (record.circle && record.circle !== 'N/A') {
                    resultHTML += `
                        <div class="result-row">
                            <span class="result-label">🌍 Circle</span>
                            <span class="result-value">${record.circle}</span>
                        </div>
                    `;
                }

                // Operator
                if (record.operator && record.operator !== 'N/A') {
                    resultHTML += `
                        <div class="result-row">
                            <span class="result-label">📡 Operator</span>
                            <span class="result-value">${record.operator}</span>
                        </div>
                    `;
                }

                // Alternate Mobile
                if (record.alt_mobile && record.alt_mobile !== 'N/A') {
                    resultHTML += `
                        <div class="result-row">
                            <span class="result-label">📞 Alt Mobile</span>
                            <span class="result-value">${record.alt_mobile}</span>
                        </div>
                    `;
                }

                // Aadhaar ID
                if (record.aadhaar_id && record.aadhaar_id !== 'N/A') {
                    resultHTML += `
                        <div class="result-row">
                            <span class="result-label">🆔 Aadhaar ID</span>
                            <span class="result-value">${record.aadhaar_id}</span>
                        </div>
                    `;
                }

                // Address (full width)
                if (record.address && record.address !== 'N/A') {
                    resultHTML += `
                        <div class="result-row" style="flex-direction: column; align-items: flex-start; padding: 16px 0;">
                            <span class="result-label" style="margin-bottom: 8px;">📍 Address</span>
                            <span class="result-value" style="text-align: left; line-height: 1.5; font-size: 13px; font-weight: 500;">${record.address}</span>
                        </div>
                    `;
                }

                // Add separator between records (except last one)
                if (index < recordCount - 1) {
                    resultHTML += `
                        <div style="height: 1px; background: linear-gradient(90deg, transparent, #e0e0e0, transparent); margin: 20px 0;"></div>
                    `;
                }

                allResultsHTML += resultHTML;
            });

            document.getElementById('resultData').innerHTML = allResultsHTML;
            resultCard.classList.add('show');
            errorCard.classList.remove('show');
        }

        function showError(title, message) {
            document.getElementById('errorTitle').textContent = title;
            document.getElementById('errorMessage').textContent = message;
            errorCard.classList.add('show');
            resultCard.classList.remove('show');
        }

        // Show toast notifications from PHP
        <?php if ($toast_success): ?>
            showToast('<?php echo addslashes($toast_success); ?>', 'success');
        <?php endif; ?>

        <?php if ($toast_error): ?>
            showToast('<?php echo addslashes($toast_error); ?>', 'error');
        <?php endif; ?>

        // Voice Search Implementation
        let recognition = null;
        let isListening = false;

        function initVoiceRecognition() {
            if ('webkitSpeechRecognition' in window || 'SpeechRecognition' in window) {
                const SpeechRecognition = window.SpeechRecognition || window.webkitSpeechRecognition;
                recognition = new SpeechRecognition();

                recognition.continuous = false;
                recognition.interimResults = false;
                recognition.lang = 'en-IN'; // Indian English

                recognition.onstart = function () {
                    isListening = true;
                    document.getElementById('voiceBtn').classList.add('listening');
                    document.getElementById('voiceStatus').style.display = 'block';
                    document.getElementById('voiceStatus').style.color = '#FF3B30';
                    document.getElementById('voiceStatus').textContent = '🎤 Listening... Speak the mobile number';
                };

                recognition.onresult = function (event) {
                    const transcript = event.results[0][0].transcript;
                    const numbers = transcript.replace(/\D/g, ''); // Extract only numbers

                    if (numbers.length === 10) {
                        document.getElementById('mobileInput').value = numbers;
                        document.getElementById('voiceStatus').style.color = '#34C759';
                        document.getElementById('voiceStatus').textContent = '✓ Number captured: ' + numbers;

                        setTimeout(() => {
                            document.getElementById('voiceStatus').style.display = 'none';
                        }, 3000);
                    } else {
                        document.getElementById('voiceStatus').style.color = '#FF9500';
                        document.getElementById('voiceStatus').textContent = '⚠️ Please speak a 10-digit number';
                    }
                };

                recognition.onerror = function (event) {
                    console.error('Speech recognition error:', event.error);
                    document.getElementById('voiceStatus').style.color = '#FF3B30';

                    switch (event.error) {
                        case 'no-speech':
                            document.getElementById('voiceStatus').textContent = '❌ No speech detected. Try again.';
                            break;
                        case 'audio-capture':
                            document.getElementById('voiceStatus').textContent = '❌ Microphone not found.';
                            break;
                        case 'not-allowed':
                            document.getElementById('voiceStatus').textContent = '❌ Microphone permission denied.';
                            break;
                        default:
                            document.getElementById('voiceStatus').textContent = '❌ Error: ' + event.error;
                    }

                    setTimeout(() => {
                        document.getElementById('voiceStatus').style.display = 'none';
                    }, 3000);
                };

                recognition.onend = function () {
                    isListening = false;
                    document.getElementById('voiceBtn').classList.remove('listening');
                };

                return true;
            } else {
                return false;
            }
        }

        function startVoiceSearch() {
            if (!recognition) {
                const supported = initVoiceRecognition();
                if (!supported) {
                    showToast('Voice search not supported in this browser. Use Chrome or Edge.', 'error');
                    return;
                }
            }

            if (isListening) {
                recognition.stop();
                return;
            }

            try {
                recognition.start();
            } catch (error) {
                console.error('Error starting recognition:', error);
                showToast('Failed to start voice search. Please try again.', 'error');
            }
        }

        // Aadhaar Search Function
        async function searchAadhaar() {
            const mobileInput = document.getElementById('mobileInput');
            if (!mobileInput) {
                console.error('mobileInput not found');
                alert('Input field not found. Please refresh the page.');
                return;
            }

            const aadhaar = mobileInput.value.trim();
            console.log('Aadhaar Search Started:', aadhaar);

            if (!aadhaar || aadhaar.length !== 12) {
                showToast('Please enter a valid 12-digit Aadhaar number', 'error');
                return;
            }

            const btn = document.getElementById('aadhaarBtn');
            const originalHTML = btn ? btn.innerHTML : '';

            if (btn) {
                btn.disabled = true;
                btn.innerHTML = '🔄 Searching...';
            }

            try {
                console.log('Fetching from API...');
                const response = await fetch('api/aadhaar_search.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/x-www-form-urlencoded',
                    },
                    body: 'aadhaar=' + encodeURIComponent(aadhaar)
                });

                console.log('Response status:', response.status);
                const text = await response.text();
                console.log('Response:', text.substring(0, 200));

                const data = JSON.parse(text);
                console.log('Success:', data.success);

                if (data.success && data.data) {
                    displayAadhaarResult(data.data);
                    showToast('Aadhaar data found!', 'success');
                } else {
                    showToast(data.error || 'No data found', 'error');
                }
            } catch (error) {
                console.error('Error:', error);
                showToast('Search failed: ' + error.message, 'error');
            } finally {
                if (btn) {
                    btn.disabled = false;
                    btn.innerHTML = originalHTML || 'Search by Aadhaar';
                }
            }
        }

        function displayAadhaarResult(data) {
            const resultCard = document.getElementById('resultCard');
            const resultContent = document.getElementById('resultContent');

            resultContent.innerHTML = `
                <div style="text-align: center; margin-bottom: 20px;">
                    <div style="width: 80px; height: 80px; margin: 0 auto 12px; border-radius: 50%; background: linear-gradient(135deg, #FF9500 0%, #FF5E3A 100%); display: flex; align-items: center; justify-content: center; font-size: 32px; color: white;">
                        ${data.photo || '👤'}
                    </div>
                    <h3 style="margin: 0; font-size: 20px; font-weight: 700;">${data.name || 'N/A'}</h3>
                    <p style="margin: 4px 0 0; color: #8E8E93; font-size: 14px;">Aadhaar Holder</p>
                </div>
                
                <div class="info-grid">
                    <div class="info-item">
                        <span class="info-label">📅 Date of Birth</span>
                        <span class="info-value">${data.dob || 'N/A'}</span>
                    </div>
                    <div class="info-item">
                        <span class="info-label">⚧ Gender</span>
                        <span class="info-value">${data.gender || 'N/A'}</span>
                    </div>
                    <div class="info-item">
                        <span class="info-label">📍 Address</span>
                        <span class="info-value">${data.address || 'N/A'}</span>
                    </div>
                </div>
            `;

            resultCard.style.display = 'block';
            resultCard.scrollIntoView({ behavior: 'smooth', block: 'nearest' });
        }

        // Aadhaar Search Function
        async function searchAadhaar() {
            const aadhaarInput = document.getElementById('aadhaarInput');
            if (!aadhaarInput) {
                showToast('Aadhaar input not found', 'error');
                return;
            }

            const aadhaar = aadhaarInput.value.trim();

            if (!aadhaar || aadhaar.length !== 12) {
                showToast('Please enter a valid 12-digit Aadhaar number', 'error');
                return;
            }

            const btn = document.getElementById('aadhaarBtn');
            if (btn) {
                btn.disabled = true;
                btn.innerHTML = '🔄 Searching...';
            }

            try {
                const response = await fetch('api/aadhaar_search.php', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
                    body: 'aadhaar=' + encodeURIComponent(aadhaar)
                });

                const data = await response.json();

                if (data.success && data.data) {
                    displayAadhaarResult(data.data);
                    showToast('Aadhaar data found!', 'success');
                } else {
                    showToast(data.error || 'No data found', 'error');
                }
            } catch (error) {
                showToast('Search failed: ' + error.message, 'error');
            } finally {
                if (btn) {
                    btn.disabled = false;
                    btn.innerHTML = `
                        <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                            <rect x="3" y="4" width="18" height="16" rx="2" />
                            <path d="M7 8h10M7 12h10M7 16h6" />
                        </svg>
                        Search Aadhaar
                    `;
                }
            }
        }

        function displayAadhaarResult(data) {
            const resultCard = document.getElementById('resultCard');
            const resultContent = document.getElementById('resultContent');

            let familyHTML = '';
            if (data.familyMembers && data.familyMembers.length > 0) {
                familyHTML = '<div style="margin-top: 20px;"><h4 style="font-size: 16px; margin-bottom: 10px;">👨‍👩‍👧‍👦 Family Members (' + data.totalMembers + ')</h4>';
                data.familyMembers.forEach(member => {
                    familyHTML += `<div style="padding: 10px; background: #f5f5f7; border-radius: 8px; margin-bottom: 8px;">
                        <strong>${member.memberName}</strong><br>
                        <small style="color: #8E8E93;">${member.releationship_name} ${member.uid === 'Yes' ? '✓ UID' : '✗'}</small>
                    </div>`;
                });
                familyHTML += '</div>';
            }

            resultContent.innerHTML = `
                <div style="text-align: center; margin-bottom: 20px;">
                    <div style="width: 80px; height: 80px; margin: 0 auto 12px; border-radius: 50%; background: linear-gradient(135deg, #FF9500 0%, #FF5E3A 100%); display: flex; align-items: center; justify-content: center; font-size: 32px; color: white;">
                        👤
                    </div>
                    <h3 style="margin: 0; font-size: 20px; font-weight: 700;">${data.name}</h3>
                    <p style="margin: 4px 0 0; color: #8E8E93; font-size: 14px;">Ration Card Holder</p>
                </div>
                <div class="info-grid">
                    <div class="info-item"><span class="info-label">📍 Address</span><span class="info-value">${data.address}</span></div>
                    <div class="info-item"><span class="info-label">🏙️ District</span><span class="info-value">${data.district}</span></div>
                    <div class="info-item"><span class="info-label">🗺️ State</span><span class="info-value">${data.state}</span></div>
                    <div class="info-item"><span class="info-label">🆔 RC ID</span><span class="info-value">${data.rcId}</span></div>
                    <div class="info-item"><span class="info-label">🏪 FPS ID</span><span class="info-value">${data.fpsId}</span></div>
                    <div class="info-item"><span class="info-label">📋 Scheme</span><span class="info-value">${data.scheme}</span></div>
                </div>
                ${familyHTML}
            `;

            resultCard.style.display = 'block';
            resultCard.scrollIntoView({ behavior: 'smooth', block: 'nearest' });
        }

        // Initialize on page load
        document.addEventListener('DOMContentLoaded', function () {
            initVoiceRecognition();
        });
    </script>

    <!-- AI Chatbot -->
    <?php include 'chat_widget.php'; ?>
</body>

</html>