<?php
/**
 * Secure Database Configuration with Prepared Statements
 * IMPROVED VERSION - Use this for production
 */

// Database credentials
define('DB_HOST', 'localhost');
define('DB_USER', 'root');
define('DB_PASS', '');
define('DB_NAME', 'mobile_app_db');

// Create connection
$conn = mysqli_connect(DB_HOST, DB_USER, DB_PASS);

// Check connection
if (!$conn) {
    die("Connection failed: " . mysqli_connect_error());
}

// Create database if not exists
$sql = "CREATE DATABASE IF NOT EXISTS " . DB_NAME;
if (!mysqli_query($conn, $sql)) {
    die("Error creating database: " . mysqli_error($conn));
}

// Select database
mysqli_select_db($conn, DB_NAME);

// Create users table if not exists
$table_sql = "CREATE TABLE IF NOT EXISTS users (
    id INT AUTO_INCREMENT PRIMARY KEY,
    name VARCHAR(100) NOT NULL,
    email VARCHAR(100) UNIQUE NOT NULL,
    mobile VARCHAR(15) UNIQUE NOT NULL,
    password VARCHAR(255) NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    INDEX idx_email (email),
    INDEX idx_mobile (mobile)
)";

if (!mysqli_query($conn, $table_sql)) {
    die("Error creating table: " . mysqli_error($conn));
}

// Create user_activities table if not exists
$activities_table_sql = "CREATE TABLE IF NOT EXISTS user_activities (
    id INT AUTO_INCREMENT PRIMARY KEY,
    user_id INT NOT NULL,
    activity_type VARCHAR(50) NOT NULL,
    activity_title VARCHAR(255) NOT NULL,
    activity_description TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    INDEX idx_user_id (user_id),
    INDEX idx_created_at (created_at),
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE
)";

if (!mysqli_query($conn, $activities_table_sql)) {
    die("Error creating activities table: " . mysqli_error($conn));
}

// Set charset to UTF-8
mysqli_set_charset($conn, "utf8mb4");

/**
 * Sanitize input data (for non-SQL contexts)
 */
function sanitize_input($data)
{
    $data = trim($data);
    $data = stripslashes($data);
    $data = htmlspecialchars($data, ENT_QUOTES, 'UTF-8');
    return $data;
}

/**
 * Check if email exists (SECURE with prepared statement)
 */
function email_exists($email)
{
    global $conn;
    $stmt = $conn->prepare("SELECT id FROM users WHERE email = ? LIMIT 1");
    $stmt->bind_param("s", $email);
    $stmt->execute();
    $result = $stmt->get_result();
    $exists = $result->num_rows > 0;
    $stmt->close();
    return $exists;
}

/**
 * Check if mobile exists (SECURE with prepared statement)
 */
function mobile_exists($mobile)
{
    global $conn;
    $stmt = $conn->prepare("SELECT id FROM users WHERE mobile = ? LIMIT 1");
    $stmt->bind_param("s", $mobile);
    $stmt->execute();
    $result = $stmt->get_result();
    $exists = $result->num_rows > 0;
    $stmt->close();
    return $exists;
}

/**
 * Validate email format
 */
function validate_email($email)
{
    return filter_var($email, FILTER_VALIDATE_EMAIL);
}

/**
 * Validate mobile number
 */
function validate_mobile($mobile)
{
    return preg_match('/^[0-9]{10,15}$/', $mobile);
}

/**
 * Log user activity (SECURE with prepared statement)
 */
function log_activity($user_id, $activity_type, $activity_title, $activity_description = '')
{
    global $conn;
    $stmt = $conn->prepare("INSERT INTO user_activities (user_id, activity_type, activity_title, activity_description) VALUES (?, ?, ?, ?)");
    $stmt->bind_param("isss", $user_id, $activity_type, $activity_title, $activity_description);
    $result = $stmt->execute();
    $stmt->close();
    return $result;
}

/**
 * Get recent activities for a user (SECURE with prepared statement)
 */
function get_recent_activities($user_id, $limit = 5)
{
    global $conn;
    $stmt = $conn->prepare("SELECT * FROM user_activities WHERE user_id = ? ORDER BY created_at DESC LIMIT ?");
    $stmt->bind_param("ii", $user_id, $limit);
    $stmt->execute();
    $result = $stmt->get_result();

    $activities = [];
    while ($row = $result->fetch_assoc()) {
        $activities[] = $row;
    }

    $stmt->close();
    return $activities;
}

/**
 * Get time ago format
 */
function time_ago($timestamp)
{
    // Handle MySQL TIMESTAMP format
    $time_ago = strtotime($timestamp);
    $current_time = time();
    $time_difference = $current_time - $time_ago;

    // Return actual date/time if timestamp is invalid
    if ($time_ago === false || $time_difference < 0) {
        return date('M d, Y h:i A', strtotime($timestamp));
    }

    $seconds = $time_difference;

    // Calculate time units
    $minutes = floor($seconds / 60);
    $hours = floor($seconds / 3600);
    $days = floor($seconds / 86400);
    $weeks = floor($seconds / 604800);
    $months = floor($seconds / 2629440);
    $years = floor($seconds / 31553280);

    // Return appropriate format
    if ($seconds < 60) {
        return "Just now";
    } else if ($minutes < 60) {
        return $minutes == 1 ? "1 minute ago" : "$minutes minutes ago";
    } else if ($hours < 24) {
        return $hours == 1 ? "1 hour ago" : "$hours hours ago";
    } else if ($days < 7) {
        return $days == 1 ? "Yesterday" : "$days days ago";
    } else if ($weeks < 4) {
        return $weeks == 1 ? "1 week ago" : "$weeks weeks ago";
    } else if ($months < 12) {
        return $months == 1 ? "1 month ago" : "$months months ago";
    } else {
        return $years == 1 ? "1 year ago" : "$years years ago";
    }
}

/**
 * Generate CSRF Token
 */
function generate_csrf_token()
{
    if (!isset($_SESSION['csrf_token'])) {
        $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
    }
    return $_SESSION['csrf_token'];
}

/**
 * Verify CSRF Token
 */
function verify_csrf_token($token)
{
    return isset($_SESSION['csrf_token']) && hash_equals($_SESSION['csrf_token'], $token);
}

/**
 * Set security headers
 */
function set_security_headers()
{
    header("X-Frame-Options: DENY");
    header("X-Content-Type-Options: nosniff");
    header("X-XSS-Protection: 1; mode=block");
    header("Referrer-Policy: strict-origin-when-cross-origin");
    header("Content-Security-Policy: default-src 'self' 'unsafe-inline' 'unsafe-eval'");
}

// Set security headers for all pages
set_security_headers();
?>