<?php
/**
 * Database Configuration
 * MySQL Connection Handler
 */

// Database credentials
define('DB_HOST', 'localhost');
define('DB_USER', 'root');
define('DB_PASS', '');
define('DB_NAME', 'mobile_app_db');

// Create connection
$conn = mysqli_connect(DB_HOST, DB_USER, DB_PASS);

// Check connection
if (!$conn) {
    die("Connection failed: " . mysqli_connect_error());
}

// Create database if not exists
$sql = "CREATE DATABASE IF NOT EXISTS " . DB_NAME;
if (!mysqli_query($conn, $sql)) {
    die("Error creating database: " . mysqli_error($conn));
}

// Select database
mysqli_select_db($conn, DB_NAME);

// Create users table
$table_sql = "CREATE TABLE IF NOT EXISTS users (
    id INT AUTO_INCREMENT PRIMARY KEY,
    name VARCHAR(100) NOT NULL,
    email VARCHAR(100) UNIQUE NOT NULL,
    mobile VARCHAR(15) UNIQUE NOT NULL,
    password VARCHAR(255) NOT NULL,
    role ENUM('user','admin') DEFAULT 'user',
    status ENUM('active','disabled') DEFAULT 'active',
    profile_picture VARCHAR(255) DEFAULT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    INDEX idx_email (email),
    INDEX idx_mobile (mobile),
    INDEX idx_role (role),
    INDEX idx_status (status)
)";

if (!mysqli_query($conn, $table_sql)) {
    die("Error creating table: " . mysqli_error($conn));
}

// Create user_activities table
$activities_table_sql = "CREATE TABLE IF NOT EXISTS user_activities (
    id INT AUTO_INCREMENT PRIMARY KEY,
    user_id INT NOT NULL,
    activity_type VARCHAR(50) NOT NULL,
    activity_title VARCHAR(255) NOT NULL,
    activity_description TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    KEY user_id (user_id),
    KEY created_at (created_at)
)";

if (!mysqli_query($conn, $activities_table_sql)) {
    die("Error creating activities table: " . mysqli_error($conn));
}

// Set charset to UTF-8
mysqli_set_charset($conn, "utf8mb4");

/**
 * Sanitize input data
 */
function sanitize_input($data)
{
    global $conn;
    $data = trim($data);
    $data = stripslashes($data);
    $data = htmlspecialchars($data);
    $data = mysqli_real_escape_string($conn, $data);
    return $data;
}

/**
 * Check if email exists
 */
function email_exists($email)
{
    global $conn;
    $email = sanitize_input($email);
    $sql = "SELECT id FROM users WHERE email = '$email'";
    $result = mysqli_query($conn, $sql);
    return mysqli_num_rows($result) > 0;
}

/**
 * Check if mobile exists
 */
function mobile_exists($mobile)
{
    global $conn;
    $mobile = sanitize_input($mobile);
    $sql = "SELECT id FROM users WHERE mobile = '$mobile'";
    $result = mysqli_query($conn, $sql);
    return mysqli_num_rows($result) > 0;
}

/**
 * Validate email format
 */
function validate_email($email)
{
    return filter_var($email, FILTER_VALIDATE_EMAIL);
}

/**
 * Validate mobile number
 */
function validate_mobile($mobile)
{
    return preg_match('/^[0-9]{10,15}$/', $mobile);
}

/**
 * Log user activity
 */
function log_activity($user_id, $activity_type, $activity_title, $activity_description = '')
{
    global $conn;
    $user_id = (int) $user_id;
    $activity_type = sanitize_input($activity_type);
    $activity_title = sanitize_input($activity_title);
    $activity_description = sanitize_input($activity_description);

    $sql = "INSERT INTO user_activities (user_id, activity_type, activity_title, activity_description) 
            VALUES ('$user_id', '$activity_type', '$activity_title', '$activity_description')";

    return mysqli_query($conn, $sql);
}

/**
 * Get recent activities for a user
 */
function get_recent_activities($user_id, $limit = 5)
{
    global $conn;
    $user_id = (int) $user_id;
    $limit = (int) $limit;

    $sql = "SELECT * FROM user_activities 
            WHERE user_id = '$user_id' 
            ORDER BY created_at DESC 
            LIMIT $limit";

    $result = mysqli_query($conn, $sql);
    $activities = [];

    while ($row = mysqli_fetch_assoc($result)) {
        $activities[] = $row;
    }

    return $activities;
}

/**
 * Get time ago format
 */
function time_ago($timestamp)
{
    $time_ago = strtotime($timestamp);
    $current_time = time();
    $time_difference = $current_time - $time_ago;

    if ($time_ago === false || $time_difference < 0) {
        return date('M d, Y h:i A', strtotime($timestamp));
    }

    $seconds = $time_difference;
    $minutes = floor($seconds / 60);
    $hours = floor($seconds / 3600);
    $days = floor($seconds / 86400);
    $weeks = floor($seconds / 604800);
    $months = floor($seconds / 2629440);
    $years = floor($seconds / 31553280);

    if ($seconds < 60) {
        return "Just now";
    } else if ($minutes < 60) {
        return $minutes == 1 ? "1 minute ago" : "$minutes minutes ago";
    } else if ($hours < 24) {
        return $hours == 1 ? "1 hour ago" : "$hours hours ago";
    } else if ($days < 7) {
        return $days == 1 ? "Yesterday" : "$days days ago";
    } else if ($weeks < 4) {
        return $weeks == 1 ? "1 week ago" : "$weeks weeks ago";
    } else if ($months < 12) {
        return $months == 1 ? "1 month ago" : "$months months ago";
    } else {
        return $years == 1 ? "1 year ago" : "$years years ago";
    }
}

/**
 * Check rate limit
 */
function check_rate_limit($email, $ip_address)
{
    global $conn;
    $email = mysqli_real_escape_string($conn, $email);
    $ip_address = mysqli_real_escape_string($conn, $ip_address);

    $sql = "SELECT COUNT(*) as count FROM login_attempts 
            WHERE (email = '$email' OR ip_address = '$ip_address') 
            AND success = 0 
            AND attempt_time > DATE_SUB(NOW(), INTERVAL 15 MINUTE)";
    $result = mysqli_query($conn, $sql);
    if ($result) {
        $count = mysqli_fetch_assoc($result)['count'];
        return $count >= 5;
    }
    return false;
}

/**
 * Log login attempt
 */
function log_login_attempt($email, $ip_address, $success = 0)
{
    global $conn;
    $email = mysqli_real_escape_string($conn, $email);
    $ip_address = mysqli_real_escape_string($conn, $ip_address);
    $success = (int) $success;

    $sql = "INSERT INTO login_attempts (email, ip_address, success) VALUES ('$email', '$ip_address', $success)";
    return mysqli_query($conn, $sql);
}
?>