<?php
session_start();
require_once '../db.php';

// Check if admin is logged in
if (!isset($_SESSION['admin_id'])) {
    header('Location: login.php');
    exit();
}

/**
 * Get all users with chat messages
 */
function get_users_with_chats()
{
    global $conn;

    $sql = "SELECT DISTINCT u.id, u.name, u.email, u.profile_picture,
            (SELECT COUNT(*) FROM chat_messages WHERE user_id = u.id AND is_user = 1 AND is_admin = 0) as unread_count,
            (SELECT created_at FROM chat_messages WHERE user_id = u.id ORDER BY created_at DESC LIMIT 1) as last_message_time
            FROM users u
            WHERE EXISTS (SELECT 1 FROM chat_messages WHERE user_id = u.id)
            ORDER BY last_message_time DESC";

    $result = mysqli_query($conn, $sql);
    $users = [];

    if ($result) {
        while ($row = mysqli_fetch_assoc($result)) {
            $users[] = $row;
        }
    }

    return $users;
}

/**
 * Get chat messages for a specific user
 */
function get_user_chat_messages($user_id)
{
    global $conn;

    $user_id = (int) $user_id;

    $sql = "SELECT cm.*, u.name as user_name 
            FROM chat_messages cm
            LEFT JOIN users u ON cm.user_id = u.id
            WHERE cm.user_id = $user_id
            ORDER BY cm.created_at ASC";

    $result = mysqli_query($conn, $sql);
    $messages = [];

    if ($result) {
        while ($row = mysqli_fetch_assoc($result)) {
            $messages[] = $row;
        }
    }

    return $messages;
}

$users_with_chats = get_users_with_chats();
$selected_user_id = $_GET['user_id'] ?? null;
$selected_user_messages = [];
$selected_user_info = null;

if ($selected_user_id) {
    $selected_user_messages = get_user_chat_messages($selected_user_id);

    // Get user info
    $user_sql = "SELECT * FROM users WHERE id = " . (int) $selected_user_id;
    $user_result = mysqli_query($conn, $user_sql);
    $selected_user_info = mysqli_fetch_assoc($user_result);
}
?>
<!DOCTYPE html>
<html lang="en">

<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Chat with Users - Admin</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        body {
            font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, sans-serif;
            background: #F5F5F7;
        }

        .chat-container {
            display: flex;
            height: 100vh;
        }

        /* Users List */
        .users-sidebar {
            width: 320px;
            background: white;
            border-right: 1px solid #E5E5EA;
            display: flex;
            flex-direction: column;
        }

        .sidebar-header {
            padding: 20px;
            border-bottom: 1px solid #E5E5EA;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
        }

        .sidebar-header h2 {
            font-size: 20px;
            font-weight: 600;
        }

        .users-list {
            flex: 1;
            overflow-y: auto;
        }

        .user-item {
            padding: 16px 20px;
            border-bottom: 1px solid #F5F5F7;
            cursor: pointer;
            transition: background 0.2s;
            display: flex;
            align-items: center;
            gap: 12px;
            text-decoration: none;
            color: inherit;
        }

        .user-item:hover {
            background: #F5F5F7;
        }

        .user-item.active {
            background: #F0F0F5;
            border-left: 3px solid #667eea;
        }

        .user-avatar {
            width: 48px;
            height: 48px;
            border-radius: 50%;
            background: #667eea;
            color: white;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 18px;
            font-weight: 600;
            flex-shrink: 0;
        }

        .user-avatar img {
            width: 100%;
            height: 100%;
            border-radius: 50%;
            object-fit: cover;
        }

        .user-info {
            flex: 1;
            min-width: 0;
        }

        .user-name {
            font-weight: 600;
            font-size: 15px;
            margin-bottom: 4px;
            white-space: nowrap;
            overflow: hidden;
            text-overflow: ellipsis;
        }

        .user-email {
            font-size: 13px;
            color: #8E8E93;
            white-space: nowrap;
            overflow: hidden;
            text-overflow: ellipsis;
        }

        .unread-badge {
            background: #FF3B30;
            color: white;
            border-radius: 12px;
            padding: 2px 8px;
            font-size: 12px;
            font-weight: 600;
        }

        /* Chat Area */
        .chat-area {
            flex: 1;
            display: flex;
            flex-direction: column;
            background: white;
        }

        .chat-header {
            padding: 20px;
            border-bottom: 1px solid #E5E5EA;
            display: flex;
            align-items: center;
            gap: 12px;
            background: white;
        }

        .chat-header h3 {
            font-size: 18px;
            font-weight: 600;
        }

        .chat-messages {
            flex: 1;
            overflow-y: auto;
            padding: 20px;
            background: #F5F5F7;
        }

        .message {
            margin-bottom: 16px;
            display: flex;
            gap: 8px;
        }

        .message.admin {
            flex-direction: row-reverse;
        }

        .message-avatar {
            width: 32px;
            height: 32px;
            border-radius: 50%;
            background: #667eea;
            color: white;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 14px;
            flex-shrink: 0;
        }

        .message.admin .message-avatar {
            background: #34C759;
        }

        .message-content {
            max-width: 60%;
        }

        .message-bubble {
            background: white;
            padding: 12px 16px;
            border-radius: 16px;
            font-size: 14px;
            line-height: 1.4;
            word-wrap: break-word;
        }

        .message.admin .message-bubble {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
        }

        .message-time {
            font-size: 11px;
            color: #8E8E93;
            margin-top: 4px;
            padding: 0 4px;
        }

        .chat-input-area {
            padding: 16px 20px;
            background: white;
            border-top: 1px solid #E5E5EA;
            display: flex;
            gap: 12px;
            align-items: center;
        }

        .chat-input {
            flex: 1;
            padding: 12px 16px;
            border: 1px solid #E5E5EA;
            border-radius: 24px;
            font-size: 14px;
            outline: none;
            font-family: inherit;
        }

        .chat-input:focus {
            border-color: #667eea;
        }

        .send-btn {
            width: 44px;
            height: 44px;
            border-radius: 50%;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            border: none;
            color: white;
            cursor: pointer;
            display: flex;
            align-items: center;
            justify-content: center;
            transition: transform 0.2s;
        }

        .send-btn:hover {
            transform: scale(1.05);
        }

        .send-btn:active {
            transform: scale(0.95);
        }

        .empty-state {
            flex: 1;
            display: flex;
            flex-direction: column;
            align-items: center;
            justify-content: center;
            color: #8E8E93;
        }

        .empty-state svg {
            width: 80px;
            height: 80px;
            margin-bottom: 16px;
            opacity: 0.5;
        }

        .empty-state h3 {
            font-size: 18px;
            margin-bottom: 8px;
        }

        .empty-state p {
            font-size: 14px;
        }
    </style>
</head>

<body>
    <div class="chat-container">
        <!-- Users Sidebar -->
        <div class="users-sidebar">
            <div class="sidebar-header">
                <h2>💬 User Chats</h2>
                <p style="font-size: 13px; opacity: 0.9; margin-top: 4px;">
                    <?php echo count($users_with_chats); ?> conversations
                </p>
            </div>

            <div class="users-list">
                <?php if (count($users_with_chats) > 0): ?>
                    <?php foreach ($users_with_chats as $user): ?>
                        <a href="?user_id=<?php echo $user['id']; ?>"
                            class="user-item <?php echo ($selected_user_id == $user['id']) ? 'active' : ''; ?>">
                            <div class="user-avatar">
                                <?php if ($user['profile_picture'] && file_exists('../' . $user['profile_picture'])): ?>
                                    <img src="../<?php echo $user['profile_picture']; ?>" alt="">
                                <?php else: ?>
                                    <?php echo strtoupper(substr($user['name'], 0, 1)); ?>
                                <?php endif; ?>
                            </div>
                            <div class="user-info">
                                <div class="user-name"><?php echo htmlspecialchars($user['name']); ?></div>
                                <div class="user-email"><?php echo htmlspecialchars($user['email']); ?></div>
                            </div>
                            <?php if ($user['unread_count'] > 0): ?>
                                <span class="unread-badge"><?php echo $user['unread_count']; ?></span>
                            <?php endif; ?>
                        </a>
                    <?php endforeach; ?>
                <?php else: ?>
                    <div style="padding: 40px 20px; text-align: center; color: #8E8E93;">
                        <p>No user chats yet</p>
                    </div>
                <?php endif; ?>
            </div>
        </div>

        <!-- Chat Area -->
        <div class="chat-area">
            <?php if ($selected_user_info): ?>
                <!-- Chat Header -->
                <div class="chat-header">
                    <div class="user-avatar">
                        <?php if ($selected_user_info['profile_picture'] && file_exists('../' . $selected_user_info['profile_picture'])): ?>
                            <img src="../<?php echo $selected_user_info['profile_picture']; ?>" alt=""
                                style="width: 100%; height: 100%; border-radius: 50%; object-fit: cover;">
                        <?php else: ?>
                            <?php echo strtoupper(substr($selected_user_info['name'], 0, 1)); ?>
                        <?php endif; ?>
                    </div>
                    <div>
                        <h3><?php echo htmlspecialchars($selected_user_info['name']); ?></h3>
                        <p style="font-size: 13px; color: #8E8E93;">
                            <?php echo htmlspecialchars($selected_user_info['email']); ?>
                        </p>
                    </div>
                </div>

                <!-- Messages -->
                <div class="chat-messages" id="chatMessages">
                    <?php foreach ($selected_user_messages as $msg): ?>
                        <div class="message <?php echo $msg['is_admin'] ? 'admin' : 'user'; ?>">
                            <div class="message-avatar">
                                <?php echo $msg['is_admin'] ? '👨‍💼' : strtoupper(substr($msg['user_name'], 0, 1)); ?>
                            </div>
                            <div class="message-content">
                                <div class="message-bubble">
                                    <?php echo nl2br(htmlspecialchars($msg['message'])); ?>
                                </div>
                                <div class="message-time">
                                    <?php echo date('g:i A', strtotime($msg['created_at'])); ?>
                                </div>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>

                <!-- Input Area -->
                <div class="chat-input-area">
                    <input type="text" class="chat-input" id="messageInput" placeholder="Type your message..."
                        onkeypress="handleEnter(event)">
                    <button class="send-btn" onclick="sendMessage()">
                        <svg width="20" height="20" viewBox="0 0 24 24" fill="currentColor">
                            <path d="M2.01 21L23 12 2.01 3 2 10l15 2-15 2z" />
                        </svg>
                    </button>
                </div>
            <?php else: ?>
                <div class="empty-state">
                    <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                        <path d="M21 15a2 2 0 0 1-2 2H7l-4 4V5a2 2 0 0 1 2-2h14a2 2 0 0 1 2 2z" />
                    </svg>
                    <h3>Select a conversation</h3>
                    <p>Choose a user from the list to start chatting</p>
                </div>
            <?php endif; ?>
        </div>
    </div>

    <script>
        const userId = <?php echo $selected_user_id ?? 'null'; ?>;
        let lastMessageId = 0;

        function handleEnter(event) {
            if (event.key === 'Enter') {
                sendMessage();
            }
        }

        async function sendMessage() {
            if (!userId) return;

            const input = document.getElementById('messageInput');
            const message = input.value.trim();

            if (!message) return;

            try {
                const formData = new FormData();
                formData.append('action', 'admin_send');
                formData.append('user_id', userId);
                formData.append('message', message);

                const response = await fetch('../api/admin_chat_api.php', {
                    method: 'POST',
                    body: formData
                });

                const data = await response.json();

                if (data.success) {
                    input.value = '';
                    // Add message immediately without reload
                    addMessageToChat(message, true);
                    // Fetch new messages
                    fetchNewMessages();
                } else {
                    alert('Failed to send message');
                }
            } catch (error) {
                console.error('Error:', error);
                alert('Connection error');
            }
        }

        function addMessageToChat(message, isAdmin = false, time = null) {
            const messagesDiv = document.getElementById('chatMessages');
            const messageDiv = document.createElement('div');
            messageDiv.className = 'message' + (isAdmin ? ' admin' : ' user');

            const displayTime = time || new Date().toLocaleTimeString('en-US', {
                hour: 'numeric',
                minute: '2-digit',
                hour12: true
            });

            messageDiv.innerHTML = `
                <div class="message-avatar">
                    ${isAdmin ? '👨‍💼' : '<?php echo strtoupper(substr($selected_user_info['name'] ?? 'U', 0, 1)); ?>'}
                </div>
                <div class="message-content">
                    <div class="message-bubble">
                        ${message.replace(/\n/g, '<br>')}
                    </div>
                    <div class="message-time">
                        ${displayTime}
                    </div>
                </div>
            `;

            messagesDiv.appendChild(messageDiv);
            messagesDiv.scrollTop = messagesDiv.scrollHeight;
        }

        async function fetchNewMessages() {
            if (!userId) return;

            try {
                const response = await fetch(`../api/admin_chat_api.php?action=get_messages&user_id=${userId}&after_id=${lastMessageId}`);
                const data = await response.json();

                if (data.success && data.messages && data.messages.length > 0) {
                    data.messages.forEach(msg => {
                        addMessageToChat(
                            msg.message,
                            msg.is_admin == 1,
                            new Date(msg.created_at).toLocaleTimeString('en-US', {
                                hour: 'numeric',
                                minute: '2-digit',
                                hour12: true
                            })
                        );
                        lastMessageId = Math.max(lastMessageId, msg.id);
                    });
                }
            } catch (error) {
                console.error('Error fetching messages:', error);
            }
        }

        // Auto-scroll to bottom on load
        const messagesDiv = document.getElementById('chatMessages');
        if (messagesDiv) {
            messagesDiv.scrollTop = messagesDiv.scrollHeight;

            // Get last message ID
            const messages = messagesDiv.querySelectorAll('.message');
            if (messages.length > 0) {
                lastMessageId = <?php echo !empty($selected_user_messages) ? end($selected_user_messages)['id'] : 0; ?>;
            }
        }

        // Real-time updates every 3 seconds
        if (userId) {
            setInterval(fetchNewMessages, 3000);
        }
    </script>
</body>

</html>