<?php
require_once __DIR__ . '/../includes/auth.php';

requireUser();

if (!hasNavPermission('search')) {
    header('Location: /app/user/dashboard.php');
    exit;
}

$user = getCurrentUser();
$permissions = getUserNavPermissions();
?>
<!DOCTYPE html>
<html lang="en">

<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0, maximum-scale=1.0, user-scalable=no">
    <meta name="theme-color" content="#FFFFFF">
    <title>Bulk Upload - Gravity</title>
    <link rel="stylesheet" href="/app/assets/css/style.css">
    <link rel="stylesheet" href="/app/assets/css/user.css">
    <style>
        .upload-card {
            background: var(--bg-primary);
            border-radius: 24px;
            padding: 24px;
            margin-bottom: 20px;
            border: 1px solid var(--separator-color);
            box-shadow: var(--shadow-sm);
        }

        .upload-area {
            border: 3px dashed var(--primary-color);
            border-radius: 20px;
            padding: 40px 20px;
            text-align: center;
            background: rgba(0, 122, 255, 0.05);
            cursor: pointer;
            transition: all 0.3s ease;
            margin-bottom: 20px;
        }

        .upload-area:hover {
            background: rgba(0, 122, 255, 0.1);
            border-color: #0051D5;
        }

        .upload-area.dragover {
            background: rgba(52, 199, 89, 0.1);
            border-color: #34C759;
        }

        .upload-icon {
            font-size: 48px;
            margin-bottom: 12px;
        }

        .progress-card {
            display: none;
            background: var(--bg-primary);
            border-radius: 20px;
            padding: 20px;
            margin-bottom: 20px;
        }

        .progress-bar {
            width: 100%;
            height: 8px;
            background: var(--separator-color);
            border-radius: 10px;
            overflow: hidden;
            margin-bottom: 12px;
        }

        .progress-fill {
            height: 100%;
            background: linear-gradient(90deg, var(--primary-color), #34C759);
            width: 0%;
            transition: width 0.3s ease;
        }

        .stats-grid {
            display: grid;
            grid-template-columns: repeat(2, 1fr);
            gap: 12px;
            margin-top: 16px;
        }

        .stat-box {
            background: rgba(0, 122, 255, 0.05);
            padding: 16px;
            border-radius: 16px;
            text-align: center;
        }

        .stat-value {
            font-size: 28px;
            font-weight: 800;
            color: var(--primary-color);
            margin-bottom: 4px;
        }

        .stat-label {
            font-size: 11px;
            color: var(--text-tertiary);
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }
    </style>
</head>

<body>
    <div class="user-container">
        <header class="app-header">
            <div class="welcome-section">
                <span class="welcome-label">Bulk Upload</span>
                <h1 class="welcome-name">Import Contacts</h1>
            </div>
            <a href="/app/user/search.php" class="profile-thumb">
                <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                    <path d="M15 18l-6-6 6-6"></path>
                </svg>
            </a>
        </header>

        <main class="app-main" style="position: relative; z-index: 1;">
            <!-- Upload Area -->
            <div class="upload-card">
                <div class="upload-area" id="uploadArea">
                    <div class="upload-icon">📤</div>
                    <h3 style="font-size: 18px; font-weight: 700; margin-bottom: 8px;">Upload VCF File</h3>
                    <p style="color: var(--text-tertiary); font-size: 14px;">Click to select or drag & drop</p>
                    <input type="file" id="fileInput" accept=".vcf" style="display: none;">
                </div>

                <button id="uploadBtn" class="btn-search-app" style="width: 100%;" disabled>
                    Start Processing
                </button>
            </div>

            <!-- Progress -->
            <div class="progress-card" id="progressCard">
                <h3 style="font-size: 16px; font-weight: 700; margin-bottom: 12px;">Processing Contacts...</h3>
                <div class="progress-bar">
                    <div class="progress-fill" id="progressFill"></div>
                </div>
                <p id="progressText" style="text-align: center; font-size: 14px; color: var(--text-secondary);">0%
                    Complete</p>

                <div class="stats-grid">
                    <div class="stat-box">
                        <div class="stat-value" id="totalStat">0</div>
                        <div class="stat-label">Total</div>
                    </div>
                    <div class="stat-box">
                        <div class="stat-value" id="processedStat">0</div>
                        <div class="stat-label">Processed</div>
                    </div>
                    <div class="stat-box">
                        <div class="stat-value" id="successStat">0</div>
                        <div class="stat-label">Success</div>
                    </div>
                    <div class="stat-box">
                        <div class="stat-value" id="failedStat">0</div>
                        <div class="stat-label">Failed</div>
                    </div>
                </div>
            </div>

            <!-- Info -->
            <div class="section">
                <h3 class="section-title">How it works</h3>
                <div
                    style="background: var(--bg-primary); padding: 20px; border-radius: 20px; border: 1px solid var(--separator-color);">
                    <p style="font-size: 14px; line-height: 1.6; color: var(--text-secondary); margin-bottom: 12px;">
                        1️⃣ Upload your contacts VCF file<br>
                        2️⃣ System extracts phone numbers<br>
                        3️⃣ Searches each number via API<br>
                        4️⃣ Stores results in your account
                    </p>
                    <p style="font-size: 12px; color: var(--text-tertiary);">
                        ⚡ Processing time: ~0.5s per contact
                    </p>
                </div>
            </div>
        </main>

        <?php include __DIR__ . '/../includes/user-nav.php'; ?>
    </div>

    <div id="toast" class="toast"></div>

    <script>
        const uploadArea = document.getElementById('uploadArea');
        const fileInput = document.getElementById('fileInput');
        const uploadBtn = document.getElementById('uploadBtn');
        const progressCard = document.getElementById('progressCard');
        let selectedFile = null;
        let processing = false;

        uploadArea.addEventListener('click', () => fileInput.click());

        fileInput.addEventListener('change', (e) => {
            selectedFile = e.target.files[0];
            if (selectedFile) {
                uploadArea.innerHTML = `
                    <div class="upload-icon">✅</div>
                    <h3 style="font-size: 18px; font-weight: 700; margin-bottom: 8px;">${selectedFile.name}</h3>
                    <p style="color: var(--text-tertiary); font-size: 14px;">${(selectedFile.size / 1024).toFixed(2)} KB</p>
                `;
                uploadBtn.disabled = false;
            }
        });

        uploadArea.addEventListener('dragover', (e) => {
            e.preventDefault();
            uploadArea.classList.add('dragover');
        });

        uploadArea.addEventListener('dragleave', () => {
            uploadArea.classList.remove('dragover');
        });

        uploadArea.addEventListener('drop', (e) => {
            e.preventDefault();
            uploadArea.classList.remove('dragover');
            const files = e.dataTransfer.files;
            if (files.length > 0 && files[0].name.endsWith('.vcf')) {
                selectedFile = files[0];
                fileInput.files = files;
                uploadArea.innerHTML = `
                    <div class="upload-icon">✅</div>
                    <h3 style="font-size: 18px; font-weight: 700; margin-bottom: 8px;">${selectedFile.name}</h3>
                    <p style="color: var(--text-tertiary); font-size: 14px;">${(selectedFile.size / 1024).toFixed(2)} KB</p>
                `;
                uploadBtn.disabled = false;
            }
        });

        uploadBtn.addEventListener('click', async () => {
            if (!selectedFile || processing) return;

            processing = true;
            uploadBtn.disabled = true;
            progressCard.style.display = 'block';

            const formData = new FormData();
            formData.append('vcf_file', selectedFile);

            try {
                const response = await fetch('/app/api/bulk-upload.php', {
                    method: 'POST',
                    body: formData
                });

                const data = await response.json();

                if (data.success) {
                    showToast('Upload successful! Processing contacts...', 'success');
                    document.getElementById('totalStat').textContent = data.total;

                    // Start processing
                    await processContacts(data.total);
                } else {
                    showToast(data.message || 'Upload failed', 'error');
                    processing = false;
                    uploadBtn.disabled = false;
                }
            } catch (error) {
                showToast('Network error: ' + error.message, 'error');
                processing = false;
                uploadBtn.disabled = false;
            }
        });

        async function processContacts(total) {
            let processed = 0;
            let success = 0;
            let failed = 0;

            while (processed < total && processing) {
                try {
                    const response = await fetch('/app/api/process-batch.php', {
                        method: 'POST',
                        headers: { 'Content-Type': 'application/json' },
                        body: JSON.stringify({ batch_size: 5 })
                    });

                    const data = await response.json();

                    if (data.success) {
                        processed += data.processed;
                        success += data.success_count;
                        failed += data.failed_count;

                        const percentage = Math.round((processed / total) * 100);
                        document.getElementById('progressFill').style.width = percentage + '%';
                        document.getElementById('progressText').textContent = percentage + '% Complete';
                        document.getElementById('processedStat').textContent = processed;
                        document.getElementById('successStat').textContent = success;
                        document.getElementById('failedStat').textContent = failed;

                        if (data.remaining === 0) {
                            showToast(`Completed! ${success} contacts processed successfully.`, 'success');
                            processing = false;
                            uploadBtn.textContent = 'Upload Another File';
                            uploadBtn.disabled = false;
                            break;
                        }

                        await new Promise(resolve => setTimeout(resolve, 1000));
                    } else {
                        throw new Error(data.message);
                    }
                } catch (error) {
                    showToast('Processing error: ' + error.message, 'error');
                    processing = false;
                    break;
                }
            }
        }

        function showToast(message, type = 'info') {
            const toast = document.getElementById('toast');
            toast.textContent = message;
            toast.className = `toast toast-${type} show`;
            setTimeout(() => toast.classList.remove('show'), 3000);
        }
    </script>
</body>

</html>