// Search functionality
let searchesToday = 0;
let dailyLimit = 0;

// Helper function to get ordinal suffix (1st, 2nd, 3rd, etc.)
function getOrdinal(n) {
    const s = ["th", "st", "nd", "rd"];
    const v = n % 100;
    return n + (s[(v - 20) % 10] || s[v] || s[0]);
}

// Load initial data
document.addEventListener('DOMContentLoaded', () => {
    loadSearchStats();
});
async function loadSearchStats() {
    try {
        const response = await fetch('/app/api/search-stats.php');
        const data = await response.json();

        if (data.success) {
            searchesToday = data.searches_today;
            dailyLimit = data.daily_limit;

            const badge = document.getElementById('remaining-badge');
            if (badge) {
                const remaining = Math.max(0, dailyLimit - searchesToday);
                badge.textContent = `${remaining} searches left`;
            }
        }
    } catch (error) {
        console.error('Failed to load stats:', error);
    }
}

// Load settings

// Search form submission
const searchForm = document.getElementById('searchForm');
const mobileInput = document.getElementById('searchInput') || document.getElementById('mobileNumber');
const searchResult = document.getElementById('searchResult');
const loadingDiv = document.getElementById('loading');
const errorDiv = document.getElementById('error');

if (searchForm) {
    searchForm.addEventListener('submit', async (e) => {
        e.preventDefault();

        const mobileNumber = mobileInput.value.trim();

        if (!mobileNumber) {
            showToast('Please enter a mobile number', 'error');
            return;
        }

        if (!/^[0-9]{10}$/.test(mobileNumber)) {
            showToast('Please enter a valid 10-digit mobile number', 'error');
            return;
        }

        if (searchesToday >= dailyLimit) {
            showToast('Daily search limit reached', 'error');
            return;
        }

        if (loadingDiv) loadingDiv.style.display = 'block';
        if (searchResult) searchResult.style.display = 'none';
        if (errorDiv) errorDiv.style.display = 'none';

        try {
            const response = await fetch('/app/api/search-mobile.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json'
                },
                body: JSON.stringify({ mobile_number: mobileNumber })
            });

            const data = await response.json();

            if (data.success) {
                displaySearchResult(data);
                searchesToday++;
                updateBadge();
                showToast('Search completed successfully', 'success');
                searchForm.reset();
            } else {
                showError(data.message || 'Search failed');
                showToast(data.message || 'Search failed', 'error');
            }
        } catch (error) {
            showError('Network error. Please try again.');
            showToast('Network error. Please try again.', 'error');
        } finally {
            if (loadingDiv) loadingDiv.style.display = 'none';
        }
    });
}

// Display search result - MULTIPLE RESULTS SUPPORT
function displaySearchResult(data) {
    let apiData = {};
    try {
        apiData = typeof data.data === 'string' ? JSON.parse(data.data) : data.data;
    } catch (e) {
        apiData = data.data || {};
    }

    console.log('API Response:', apiData);

    let results = [];

    if (Array.isArray(apiData)) {
        results = apiData;
    } else if (apiData.results && Array.isArray(apiData.results)) {
        results = apiData.results;
    } else if (apiData.data && Array.isArray(apiData.data)) {
        results = apiData.data;
    } else {
        results = [apiData];
    }

    const sourceBadge = `<span class="badge badge-${data.source.toLowerCase()}">${data.source}</span>`;

    let resultHTML = '';

    results.forEach((result, index) => {
        const resultNumber = results.length > 1 ? index + 1 : null;
        resultHTML += createResultCard(result, data.mobile_number, sourceBadge, resultNumber);
    });

    searchResult.innerHTML = resultHTML;
    searchResult.style.display = 'block';
    searchResult.scrollIntoView({ behavior: 'smooth', block: 'start' });
}

// Create result card with SPECIFIC FIELD ORDER
function createResultCard(apiData, mobileNumber, sourceBadge, resultNumber) {
    const name = apiData.name || apiData.Name || apiData.full_name || apiData.user || apiData.User || 'Unknown User';
    const initial = name.charAt(0).toUpperCase();

    // Create ordinal badge (1st, 2nd, 3rd, etc.)
    let numberBadge = '';
    if (resultNumber) {
        const ordinal = getOrdinal(resultNumber);
        const emoji = resultNumber === 1 ? '🥇 ' : resultNumber === 2 ? '🥈 ' : resultNumber === 3 ? '🥉 ' : '';
        numberBadge = `<span class="badge" style="background: rgba(88, 86, 214, 0.15); color: #5856D6; margin-right: 8px;">${emoji}${ordinal}</span>`;
    }

    let cardHTML = `
        <div class="result-card" style="margin-bottom: 20px;">
            <div class="result-header">
                <div class="result-avatar">${initial}</div>
                <div class="result-main-info">
                    <h3 class="result-title">${escapeHtml(name)}</h3>
                    <p class="result-subtitle">${escapeHtml(mobileNumber)}</p>
                    <div style="margin-top: 8px;">${numberBadge}${sourceBadge}</div>
                </div>
            </div>
            <div class="result-body">
    `;

    // SPECIFIC ORDER: Mobile, Name, Aadhaar, Father Name, Email, Circle, Alternative Number, Address
    const fieldOrder = [
        { keys: ['mobile', 'mobile_number', 'Mobile'], label: 'Mobile Number', emoji: '📱' },
        { keys: ['name', 'Name', 'full_name'], label: 'Name', emoji: '👤' },
        { keys: ['id_number', 'aadhaar_id', 'aadhaarid', 'Aadhaar_ID', 'aadhaar', 'aadhar'], label: 'Aadhaar Number', emoji: '🪪' },
        { keys: ['father_name', 'fathername', 'Father_Name'], label: 'Father Name', emoji: '👨' },
        { keys: ['email', 'Email'], label: 'Email', emoji: '📧' },
        { keys: ['circle', 'Circle'], label: 'Circle', emoji: '📍' },
        { keys: ['operator', 'Operator'], label: 'Operator', emoji: '📱' },
        { keys: ['alt_mobile', 'alternative_number', 'alternativenumber', 'alt_number'], label: 'Alternative Number', emoji: '☎️' },
        { keys: ['address', 'Address'], label: 'Address', emoji: '🏠' }
    ];

    const displayedKeys = new Set(['name', 'Name', 'full_name', 'user', 'User']);

    // Display in order
    let aadhaarNumber = null;

    fieldOrder.forEach(fieldDef => {
        for (const key of fieldDef.keys) {
            if (apiData.hasOwnProperty(key) && apiData[key] !== null && apiData[key] !== '') {
                if (displayedKeys.has(key)) {
                    fieldDef.keys.forEach(k => displayedKeys.add(k));
                    break;
                }

                // Store Aadhaar number for family button
                if (fieldDef.label === 'Aadhaar Number') {
                    aadhaarNumber = apiData[key];
                }

                // Add inline button for Aadhaar
                let valueContent = escapeHtml(String(apiData[key]));
                if (fieldDef.label === 'Aadhaar Number' && apiData[key].length === 12) {
                    valueContent += ` <button onclick="viewFamilyDetails('${apiData[key]}')" style="margin-left: 8px; padding: 4px 10px; background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; border: none; border-radius: 6px; font-size: 11px; font-weight: 600; cursor: pointer;">👨‍👩‍👧‍👦 Family</button>`;
                }

                // Special handling for Aadhaar - use UIDAI logo
                let iconContent = fieldDef.emoji;
                if (fieldDef.label === 'Aadhaar Number') {
                    iconContent = '<img src="/app/assets/images/uidai-logo.png" alt="UIDAI" style="width: 20px; height: 20px; object-fit: contain;">';
                }

                cardHTML += `
                    <div class="result-item">
                        <div class="result-item-icon" style="font-size: 20px;">${iconContent}</div>
                        <div class="result-item-content">
                            <span class="result-label">${fieldDef.label}</span>
                            <span class="result-value">${valueContent}</span>
                        </div>
                    </div>
                `;

                fieldDef.keys.forEach(k => displayedKeys.add(k));
                break;
            }
        }
    });



    // Remaining fields
    for (const [key, value] of Object.entries(apiData)) {
        if (!displayedKeys.has(key) && typeof value !== 'object' && value !== null && value !== '') {
            cardHTML += `
                <div class="result-item">
                    <div class="result-item-icon" style="font-size: 20px;">•</div>
                    <div class="result-item-content">
                        <span class="result-label">${formatLabel(key)}</span>
                        <span class="result-value">${escapeHtml(String(value))}</span>
                    </div>
                </div>
            `;
        }
    }

    cardHTML += `</div></div>`;
    return cardHTML;
}

function updateBadge() {
    const badge = document.getElementById('remaining-badge');
    if (badge) {
        const remaining = Math.max(0, dailyLimit - searchesToday);
        badge.textContent = `${remaining} searches left`;
    }
}

function showError(message) {
    if (errorDiv) {
        const errorMessage = document.getElementById('errorMessage');
        if (errorMessage) errorMessage.textContent = message;
        errorDiv.style.display = 'flex';
    }
}

// View Family Details - Fetch Ration Card Data
async function viewFamilyDetails(aadhaarNumber) {
    const searchResult = document.getElementById('searchResult');
    const loadingDiv = document.getElementById('loading');

    if (!aadhaarNumber || aadhaarNumber.length !== 12) {
        showToast('Invalid Aadhaar number', 'error');
        return;
    }

    try {
        if (loadingDiv) loadingDiv.style.display = 'flex';

        const response = await fetch('/app/api/search-ration.php', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ aadhaar: aadhaarNumber })
        });

        const data = await response.json();

        if (loadingDiv) loadingDiv.style.display = 'none';

        if (data.success) {
            // Append family details below mobile results
            appendFamilyDetails(data);
            showToast('Family details loaded successfully', 'success');
        } else {
            showToast(data.message || 'Failed to fetch family details', 'error');
        }
    } catch (error) {
        if (loadingDiv) loadingDiv.style.display = 'none';
        showToast('Network error. Please try again.', 'error');
        console.error('Family details error:', error);
    }
}

// Append family details to existing results
function appendFamilyDetails(data) {
    const searchResult = document.getElementById('searchResult');
    if (!searchResult) return;

    try {
        let result;
        if (data.data && data.data.result && data.data.result.results) {
            result = data.data.result.results[0];
        } else if (data.result && data.result.results) {
            result = data.result.results[0];
        } else {
            showToast('Error: Invalid data structure', 'error');
            return;
        }

        if (!result || !result.ration_card_details) {
            showToast('Error: Missing ration card details', 'error');
            return;
        }

        const rationCard = result.ration_card_details;
        const members = result.members || [];
        const additionalInfo = result.additional_info || {};

        const sourceBadge = data.source === 'database'
            ? '<span class="badge" style="background: rgba(52, 199, 89, 0.15); color: #34C759;">💾 Database</span>'
            : '<span class="badge" style="background: rgba(0, 122, 255, 0.15); color: #007AFF;">📱 API</span>';

        const familyHTML = `
        <div class="result-card" style="margin-top: 20px; border-top: 3px solid #667eea;">
            <div class="result-header">
                <div class="result-avatar" style="background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);">🎫</div>
                <div class="result-main-info">
                    <h3 class="result-title">Family Details (Ration Card)</h3>
                    <p class="result-subtitle">${rationCard.ration_card_no}</p>
                    <div style="margin-top: 8px;">${sourceBadge}</div>
                </div>
            </div>
            
            <div class="result-body">
                <div class="result-item">
                    <span class="result-label">State</span>
                    <span class="result-value">${rationCard.state_name}</span>
                </div>
                <div class="result-item">
                    <span class="result-label">District</span>
                    <span class="result-value">${rationCard.district_name}</span>
                </div>
                <div class="result-item">
                    <span class="result-label">Scheme</span>
                    <span class="result-value">${rationCard.scheme_name}</span>
                </div>
                <div class="result-item">
                    <span class="result-label">FPS Category</span>
                    <span class="result-value">${additionalInfo.fps_category || 'N/A'}</span>
                </div>
                <div class="result-item">
                    <span class="result-label">IMPDS Transaction</span>
                    <span class="result-value">
                        ${additionalInfo.impds_transaction_allowed
                ? '<span style="color: #34C759;">✅ Allowed</span>'
                : '<span style="color: #FF3B30;">❌ Not Allowed</span>'}
                    </span>
                </div>
                <div class="result-item">
                    <span class="result-label">Central Repository</span>
                    <span class="result-value">
                        ${additionalInfo.exists_in_central_repository
                ? '<span style="color: #34C759;">✅ Exists</span>'
                : '<span style="color: #FF9500;">⚠️ Not Found</span>'}
                    </span>
                </div>
                <div class="result-item">
                    <span class="result-label">Duplicate Aadhaar</span>
                    <span class="result-value">
                        ${additionalInfo.duplicate_aadhaar_beneficiary
                ? '<span style="color: #FF3B30;">⚠️ Duplicate Found</span>'
                : '<span style="color: #34C759;">✅ No Duplicate</span>'}
                    </span>
                </div>
            </div>
            
            <div style="margin-top: 20px;">
                <h4 style="font-size: 16px; font-weight: 700; margin-bottom: 12px; color: var(--text-primary);">
                    Family Members (${members.length})
                </h4>
                <div style="display: grid; gap: 8px;">
                    ${members.map(member => `
                        <div style="padding: 12px; background: var(--bg-secondary); border-radius: 8px; display: flex; justify-content: space-between; align-items: center;">
                            <div>
                                <div style="font-weight: 600; color: var(--text-primary);">${member.member_name || 'Unknown'}</div>
                                <div style="font-size: 13px; color: var(--text-tertiary); margin-top: 2px;">ID: ${member.member_id || 'N/A'}</div>
                            </div>
                            ${member.relation ? `
                                <div style="background: rgba(88, 86, 214, 0.15); color: #5856D6; padding: 4px 12px; border-radius: 12px; font-size: 12px; font-weight: 600;">
                                    ${member.relation}
                                </div>
                            ` : ''}
                        </div>
                    `).join('')}
                </div>
            </div>
        </div>
        `;

        // Append to existing results
        searchResult.insertAdjacentHTML('beforeend', familyHTML);

        // Scroll to family details
        const familyCard = searchResult.lastElementChild;
        if (familyCard) {
            familyCard.scrollIntoView({ behavior: 'smooth', block: 'start' });
        }
    } catch (error) {
        console.error('Error displaying family details:', error);
        showToast('Error displaying family details', 'error');
    }
}

function formatLabel(key) {
    return key.replace(/_/g, ' ').replace(/\b\w/g, l => l.toUpperCase());
}

function escapeHtml(text) {
    const map = { '&': '&amp;', '<': '&lt;', '>': '&gt;', '"': '&quot;', "'": '&#039;' };
    return text.replace(/[&<>"']/g, m => map[m]);
}

function showToast(message, type = 'info') {
    const toast = document.getElementById('toast');
    if (toast) {
        toast.textContent = message;
        toast.className = `toast toast-${type} show`;
        setTimeout(() => toast.classList.remove('show'), 3000);
    }
}
