// Ration Card Search Handler - Runs BEFORE search.js
(function () {
    let formHandled = false;

    document.addEventListener('DOMContentLoaded', () => {
        const searchTypeSelect = document.getElementById('searchType');
        const searchInput = document.getElementById('searchInput');
        const searchForm = document.getElementById('searchForm');

        if (!searchTypeSelect || !searchInput || !searchForm) return;

        // Update input based on search type
        searchTypeSelect.addEventListener('change', () => {
            const searchType = searchTypeSelect.value;

            if (searchType === 'mobile') {
                searchInput.setAttribute('placeholder', 'Enter mobile number...');
                searchInput.setAttribute('maxlength', '10');
                searchInput.setAttribute('pattern', '[0-9]{10}');
            } else {
                searchInput.setAttribute('placeholder', 'Enter Aadhaar number (12 digits)...');
                searchInput.setAttribute('maxlength', '12');
                searchInput.setAttribute('pattern', '[0-9]{12}');
            }

            searchInput.value = '';
            const searchResult = document.getElementById('searchResult');
            if (searchResult) searchResult.style.display = 'none';
        });

        // Intercept form submit - HIGH PRIORITY
        searchForm.addEventListener('submit', async (e) => {
            const searchType = searchTypeSelect.value;

            if (searchType === 'ration') {
                // Handle ration card search
                e.preventDefault();
                e.stopImmediatePropagation();

                const searchValue = searchInput.value.trim();
                console.log('Ration card search triggered:', searchValue);
                await handleRationCardSearch(searchValue);
                formHandled = true;
            } else {
                // Let search.js handle mobile search
                formHandled = false;
            }
        }, true); // Use capture phase to run before search.js
    });
})();

async function handleRationCardSearch(aadhaar) {
    const loadingDiv = document.getElementById('loading');
    const searchResult = document.getElementById('searchResult');
    const errorDiv = document.getElementById('error');

    // Validate
    if (!/^[0-9]{12}$/.test(aadhaar)) {
        showToast('Please enter a valid 12-digit Aadhaar number', 'error');
        return;
    }

    // Show loading
    if (loadingDiv) loadingDiv.style.display = 'block';
    if (searchResult) searchResult.style.display = 'none';
    if (errorDiv) errorDiv.style.display = 'none';

    console.log('Sending Aadhaar:', aadhaar);

    try {
        const payload = { aadhaar: aadhaar };
        console.log('Payload:', payload);

        const response = await fetch('/app/api/search-ration.php', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify(payload)
        });

        console.log('Response status:', response.status);
        const data = await response.json();
        console.log('Response data:', data);

        if (loadingDiv) loadingDiv.style.display = 'none';

        if (data.success) {
            console.log('Success! Calling displayRationCardResult...');
            displayRationCardResult(data);
            showToast('Search completed successfully', 'success');
            if (typeof loadSearchStats === 'function') loadSearchStats();
        } else {
            console.error('Search failed:', data.message);
            showToast(data.message || 'Search failed', 'error');
        }
    } catch (error) {
        if (loadingDiv) loadingDiv.style.display = 'none';
        showToast('Network error. Please try again.', 'error');
        console.error('Ration card search error:', error);
    }
}

function displayRationCardResult(data) {
    const searchResult = document.getElementById('searchResult');
    if (!searchResult) return;

    console.log('Displaying result:', data);

    try {
        // Handle different response structures
        let result;

        // Check the actual structure from API response
        if (data.data && data.data.result && data.data.result.results) {
            // From API (wrapped in data.data)
            result = data.data.result.results[0];
        } else if (data.result && data.result.results) {
            // From database (direct structure)
            result = data.result.results[0];
        } else {
            console.error('Unexpected data structure:', data);
            showToast('Error: Invalid data structure', 'error');
            return;
        }

        if (!result || !result.ration_card_details) {
            console.error('Missing ration card details:', result);
            showToast('Error: Missing ration card details', 'error');
            return;
        }

        const rationCard = result.ration_card_details;
        const members = result.members || [];
        const additionalInfo = result.additional_info || {};

        const sourceBadge = data.source === 'database'
            ? '<span class="badge" style="background: rgba(52, 199, 89, 0.15); color: #34C759;">💾 Database</span>'
            : '<span class="badge" style="background: rgba(0, 122, 255, 0.15); color: #007AFF;">📱 API</span>';

        let html = `
        <div class="result-card" style="margin-bottom: 20px;">
            <div class="result-header">
                <div class="result-avatar" style="background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);">🎫</div>
                <div class="result-main-info">
                    <h3 class="result-title">Ration Card Details</h3>
                    <p class="result-subtitle">${rationCard.ration_card_no}</p>
                    <div style="margin-top: 8px;">${sourceBadge}</div>
                </div>
            </div>
            
            <div class="result-body">
                <div class="result-item">
                    <span class="result-label">State</span>
                    <span class="result-value">${rationCard.state_name}</span>
                </div>
                <div class="result-item">
                    <span class="result-label">District</span>
                    <span class="result-value">${rationCard.district_name}</span>
                </div>
                <div class="result-item">
                    <span class="result-label">Scheme</span>
                    <span class="result-value">${rationCard.scheme_name}</span>
                </div>
                <div class="result-item">
                    <span class="result-label">FPS Category</span>
                    <span class="result-value">${additionalInfo.fps_category || 'N/A'}</span>
                </div>
                <div class="result-item">
                    <span class="result-label">IMPDS Transaction</span>
                    <span class="result-value">
                        ${additionalInfo.impds_transaction_allowed
                ? '<span style="color: #34C759;">✅ Allowed</span>'
                : '<span style="color: #FF3B30;">❌ Not Allowed</span>'}
                    </span>
                </div>
                <div class="result-item">
                    <span class="result-label">Central Repository</span>
                    <span class="result-value">
                        ${additionalInfo.exists_in_central_repository
                ? '<span style="color: #34C759;">✅ Exists</span>'
                : '<span style="color: #FF9500;">⚠️ Not Found</span>'}
                    </span>
                </div>
                <div class="result-item">
                    <span class="result-label">Duplicate Aadhaar</span>
                    <span class="result-value">
                        ${additionalInfo.duplicate_aadhaar_beneficiary
                ? '<span style="color: #FF3B30;">⚠️ Duplicate Found</span>'
                : '<span style="color: #34C759;">✅ No Duplicate</span>'}
                    </span>
                </div>
            </div>
            
            <div style="margin-top: 20px;">
                <h4 style="font-size: 16px; font-weight: 700; margin-bottom: 12px; color: var(--text-primary);">
                    Family Members (${members.length})
                </h4>
                <div style="display: grid; gap: 8px;">
                    ${members.map(member => `
                        <div style="padding: 12px; background: var(--bg-secondary); border-radius: 8px; display: flex; justify-content: space-between; align-items: center;">
                            <div>
                                <div style="font-weight: 600; color: var(--text-primary);">${member.member_name || 'Unknown'}</div>
                                <div style="font-size: 13px; color: var(--text-tertiary); margin-top: 2px;">ID: ${member.member_id || 'N/A'}</div>
                            </div>
                            ${member.relation || member.s_no ? `
                                <div style="background: rgba(88, 86, 214, 0.15); color: #5856D6; padding: 4px 12px; border-radius: 12px; font-size: 12px; font-weight: 600;">
                                    ${member.relation || '#' + member.s_no}
                                </div>
                            ` : ''}
                        </div>
                    `).join('')}
                </div>
            </div>
        </div>
    `;

        searchResult.innerHTML = html;
        searchResult.style.display = 'block';
        searchResult.scrollIntoView({ behavior: 'smooth', block: 'start' });
    } catch (error) {
        console.error('Error displaying result:', error);
        showToast('Error displaying results', 'error');
    }
}
