<?php
/**
 * Bulk Upload API - VCF File Handler
 */

ob_start();
require_once __DIR__ . '/../includes/auth.php';
ob_end_clean();

requireUser();

header('Content-Type: application/json');

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    echo json_encode(['success' => false, 'message' => 'Invalid request method']);
    exit;
}

if (!isset($_FILES['vcf_file'])) {
    echo json_encode(['success' => false, 'message' => 'No file uploaded']);
    exit;
}

$file = $_FILES['vcf_file'];
$userId = (int) $_SESSION['user_id'];

// Validate file
if ($file['error'] !== UPLOAD_ERR_OK) {
    echo json_encode(['success' => false, 'message' => 'File upload error']);
    exit;
}

if (!str_ends_with(strtolower($file['name']), '.vcf')) {
    echo json_encode(['success' => false, 'message' => 'Only .vcf files are allowed']);
    exit;
}

// Read and parse VCF
$content = file_get_contents($file['tmp_name']);
$contacts = parseVCF($content);

if (empty($contacts)) {
    echo json_encode(['success' => false, 'message' => 'No valid contacts found in file']);
    exit;
}

// Store contacts temporarily for processing
$conn = getDBConnection();

// Create temp table for this user's upload
$tempTable = "temp_upload_" . $userId;
$conn->query("DROP TABLE IF EXISTS $tempTable");
$conn->query("CREATE TABLE $tempTable (
    id INT AUTO_INCREMENT PRIMARY KEY,
    phone VARCHAR(20),
    name VARCHAR(255),
    processed TINYINT DEFAULT 0
)");

// Insert contacts
$inserted = 0;
foreach ($contacts as $contact) {
    foreach ($contact['phones'] as $phone) {
        $phone = cleanPhone($phone);
        if (strlen($phone) == 10) {
            $name = $conn->real_escape_string($contact['name']);
            $conn->query("INSERT INTO $tempTable (phone, name) VALUES ('$phone', '$name')");
            $inserted++;
        }
    }
}

$conn->close();

echo json_encode([
    'success' => true,
    'message' => "Uploaded $inserted contacts",
    'total' => $inserted
]);

function parseVCF($content)
{
    $contacts = [];
    $lines = explode("\n", $content);
    $currentContact = null;

    foreach ($lines as $line) {
        $line = trim($line);
        if (empty($line))
            continue;

        if (stripos($line, 'BEGIN:VCARD') === 0) {
            $currentContact = ['name' => '', 'phones' => []];
        } elseif (stripos($line, 'END:VCARD') === 0) {
            if ($currentContact && !empty($currentContact['phones'])) {
                $contacts[] = $currentContact;
            }
            $currentContact = null;
        } elseif ($currentContact) {
            if (stripos($line, 'FN:') === 0) {
                $currentContact['name'] = substr($line, 3);
            } elseif (stripos($line, 'N:') === 0 && empty($currentContact['name'])) {
                $parts = explode(';', substr($line, 2));
                $currentContact['name'] = trim($parts[1] . ' ' . $parts[0]);
            } elseif (stripos($line, 'TEL') === 0) {
                $phone = extractValue($line);
                if ($phone) {
                    $currentContact['phones'][] = $phone;
                }
            }
        }
    }

    return $contacts;
}

function extractValue($line)
{
    $parts = explode(':', $line, 2);
    return isset($parts[1]) ? trim($parts[1]) : '';
}

function cleanPhone($phone)
{
    $phone = preg_replace('/[^0-9+]/', '', $phone);
    $phone = preg_replace('/^\+91/', '', $phone);
    $phone = preg_replace('/^91/', '', $phone);
    if (strlen($phone) > 10) {
        $phone = substr($phone, -10);
    }
    return $phone;
}
